// content.js

if (!window.location.href.includes('/res/')) {
    console.log("🛡️ Anti-Bot Blocker: スレ一覧ページのため動作を待機します。");
} else {
    console.log("🛡️ Anti-Bot Blocker (v1.29 - Optimized) がスレッド内で起動しました！");

    // ==========================================
    // UI・スタイルの初期設定 (変更なし)
    // ==========================================
    const style = document.createElement('style');
    style.innerHTML = `
        .antibot-bot-post {
            opacity: 0.7;
            background-color: rgba(255, 0, 0, 0.05);
            border-left: 4px dashed rgba(255, 0, 0, 0.3);
            transition: opacity 0.2s ease-in-out;
        }
        .antibot-bot-post:hover { opacity: 1.0; }
        body.ab-mode-hidden .antibot-bot-post { display: none !important; }
        body.ab-mode-normal .antibot-bot-post { opacity: 1.0 !important; background-color: transparent !important; border-left: none !important; }
        body.ab-mode-normal .antibot-warning-text { display: none !important; }
        #ab-panel {
            position: fixed; bottom: 15px; right: 15px; z-index: 99999;
            background: rgba(250, 250, 250, 0.95); border: 1px solid #aaa;
            border-radius: 6px; padding: 0 10px 10px 10px; font-size: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.2); width: 200px;
            font-family: sans-serif; color: #333; transition: width 0.2s;
        }
        #ab-panel.ab-minimized { width: 100px; padding-bottom: 5px; }
        #ab-panel.ab-minimized #ab-content-full { display: none; }
        #ab-panel:not(.ab-minimized) #ab-content-mini { display: none; }
        #ab-drag-handle {
            width: 100%; height: 16px; margin: 0 0 8px 0; background-color: #ddd;
            border-radius: 5px 5px 0 0; cursor: grab; position: relative;
            margin-left: -10px; padding-right: 20px;
        }
        #ab-drag-handle:active { cursor: grabbing; }
        #ab-drag-handle::after {
            content: ""; position: absolute; top: 6px; left: 50%; transform: translateX(-50%);
            width: 30px; height: 4px; border-top: 1px solid #999; border-bottom: 1px solid #999;
        }
        #ab-toggle-size {
            position: absolute; right: 0; top: -1px; font-size: 14px; font-weight: bold;
            color: #666; cursor: pointer; padding: 0 5px; user-select: none;
        }
        #ab-toggle-size:hover { color: #000; }
        #ab-content-mini {
            text-align: center; font-weight: bold; font-size: 13px;
            padding: 5px; border-radius: 4px; color: #333; cursor: pointer;
        }
        #ab-content-mini:hover { background-color: #eee; }
        #ab-content-mini.ab-alert {
            background-color: rgba(255, 204, 204, 0.9); color: #cc0000; border: 1px solid #cc0000;
            animation: ab-blink 1.5s infinite alternate;
        }
        @keyframes ab-blink {
            from { background-color: rgba(255, 204, 204, 0.9); box-shadow: 0 0 5px #cc0000; }
            to { background-color: rgba(255, 0, 0, 0.05); box-shadow: none; }
        }
        .ab-row { margin-bottom: 6px; }
        .ab-row:last-child { margin-bottom: 0; }
        .ab-btn {
            width: 100%; padding: 5px; font-weight: bold; font-size: 11px;
            border-radius: 4px; cursor: pointer; border: 1px solid #999; background: #eee;
        }
        .ab-btn:disabled { cursor: not-allowed; opacity: 0.5; }
        .ab-btn:not(:disabled):hover { background: #ddd; }
        #ab-start-btn:not(:disabled) { background: #f90; color: #fff; border-color: #c60; }
        #ab-start-btn.stop-mode { background: #c00 !important; color: #fff !important; border-color: #900 !important; }
        #ab-cache-btn { background: #e8f4f8; border-color: #9cc; color: #369; }
        #ab-import-btn { background: #f4f8e8; border-color: #9c9; color: #363; }
        #ab-start-no { width: 100%; padding: 3px; box-sizing: border-box; font-size: 11px; }
        #ab-progress { font-size: 11px; color: #555; background: #e0e0e0; padding: 4px; border-radius: 3px; min-height: 14px; text-align: center; }
        .ab-view-btn {
            flex: 1; padding: 5px 0; font-size: 11px; font-weight: bold; border: 1px solid #ccc;
            border-radius: 4px; cursor: pointer; background: #fafafa; color: #888; transition: all 0.2s;
        }
        .ab-view-btn:hover { background: #e0e0e0; }
        .ab-view-btn.active[data-mode="normal"] { background: #e0e0e0; color: #333; border-color: #999; }
        .ab-view-btn.active[data-mode="faded"] { background: #e8f4f8; color: #369; border-color: #9cc; }
        .ab-view-btn.active[data-mode="hidden"] { background: #ffdddd; color: #c00; border-color: #c00; }
        .ab-ravaged { color: #cc0000; font-weight: bold; }
    `;
    document.head.appendChild(style);

    const panel = document.createElement('div');
    panel.id = 'ab-panel';
    panel.innerHTML = `
        <div id="ab-drag-handle" title="ドラッグして移動">
            <span id="ab-toggle-size" title="最小化/展開">[-]</span>
        </div>
        
        <div id="ab-content-full">
            <div class="ab-row" style="text-align: center; font-size: 11px; color: #555; margin-bottom: 4px;">👁️ 表示モード</div>
            <div class="ab-row" style="display: flex; gap: 4px; margin-bottom: 10px;">
                <button class="ab-view-btn" data-mode="normal" title="見た目を弄らない">通常</button>
                <button class="ab-view-btn" data-mode="faded" title="少し薄く赤みがかる">薄く</button>
                <button class="ab-view-btn" data-mode="hidden" title="完全に非表示にする">隠す</button>
            </div>
            <div class="ab-row" id="ab-stats">総:0 | Bot:0 (最大0%)</div>
            <div class="ab-row">
                <input type="text" id="ab-start-no" placeholder="開始(4桁以下なら書き込み順)">
            </div>
            <div class="ab-row">
                <button id="ab-start-btn" class="ab-btn" disabled>Del送信 (区間40%~)</button>
            </div>
            <div class="ab-row" style="display: flex; gap: 4px;">
                <button id="ab-cache-btn" class="ab-btn" title="今見ているスレからコピペを学習">今スレ学習</button>
                <button id="ab-import-btn" class="ab-btn" title="保存した過去ログ(HTML)を読み込んでコピペを学習">📂ログ読込</button>
            </div>
            <div class="ab-row" id="ab-progress">待機中</div>
            <div class="ab-row" style="display: flex; gap: 4px; justify-content: space-between; align-items: center; font-size: 10px; margin-top: 4px; border-top: 1px solid #ccc; padding-top: 4px;">
                <span title="Del送信を有効にするBot率">⚙️ Del閾値: <input type="number" id="ab-del-threshold" value="40" style="width: 35px; font-size: 10px; padding: 1px;">%</span>
                <span title="短文コピペと判定する回数">短文: <input type="number" id="ab-short-threshold" value="4" style="width: 30px; font-size: 10px; padding: 1px;">回</span>
            </div>
        </div>
        
        <div id="ab-content-mini" title="クリックで展開">Bot: 0%</div>
    `;
    document.body.appendChild(panel);

    // ==========================================
    // UIイベント管理 (変更なし)
    // ==========================================
    const toggleSizeBtn = document.getElementById('ab-toggle-size');
    let panelSize = localStorage.getItem('antibot-panel-size') || 'full';
    
    if (panelSize === 'mini') {
        panel.classList.add('ab-minimized');
        toggleSizeBtn.innerText = '[+]';
    }

    toggleSizeBtn.addEventListener('mousedown', (e) => { e.stopPropagation(); });

    function togglePanelSize() {
        if (panel.classList.contains('ab-minimized')) {
            panel.classList.remove('ab-minimized');
            toggleSizeBtn.innerText = '[-]';
            localStorage.setItem('antibot-panel-size', 'full');
        } else {
            panel.classList.add('ab-minimized');
            toggleSizeBtn.innerText = '[+]';
            localStorage.setItem('antibot-panel-size', 'mini');
        }
    }

    toggleSizeBtn.addEventListener('click', togglePanelSize);
    document.getElementById('ab-content-mini').addEventListener('click', () => {
        if (panel.classList.contains('ab-minimized')) togglePanelSize();
    });

    let currentMode = localStorage.getItem('antibot-view-mode') || 'faded';
    function applyViewMode(mode) {
        document.body.classList.remove('ab-mode-normal', 'ab-mode-hidden', 'ab-mode-faded');
        document.body.classList.add(`ab-mode-${mode}`);
        document.querySelectorAll('.ab-view-btn').forEach(btn => {
            if (btn.dataset.mode === mode) btn.classList.add('active');
            else btn.classList.remove('active');
        });
        localStorage.setItem('antibot-view-mode', mode);
        currentMode = mode;
    }
    document.querySelectorAll('.ab-view-btn').forEach(btn => {
        btn.addEventListener('click', (e) => applyViewMode(e.target.dataset.mode));
    });
    applyViewMode(currentMode);

    let DEL_THRESHOLD = 40;
    let SHORT_TEXT_THRESHOLD = 4;
    document.getElementById('ab-del-threshold').addEventListener('change', (e) => {
        DEL_THRESHOLD = parseInt(e.target.value, 10) || 40;
        queueUpdateStats();
    });
    document.getElementById('ab-short-threshold').addEventListener('change', (e) => {
        SHORT_TEXT_THRESHOLD = parseInt(e.target.value, 10) || 4;
    });

    const dragHandle = document.getElementById('ab-drag-handle');
    let isDragging = false, startX, startY, initialLeft, initialTop;
    const savedPos = JSON.parse(localStorage.getItem('antibot-panel-pos'));
    if (savedPos && savedPos.left && savedPos.top) {
        panel.style.bottom = 'auto'; panel.style.right = 'auto';
        panel.style.left = savedPos.left; panel.style.top = savedPos.top;
    }
    dragHandle.addEventListener('mousedown', (e) => {
        isDragging = true; startX = e.clientX; startY = e.clientY;
        const rect = panel.getBoundingClientRect();
        initialLeft = rect.left; initialTop = rect.top; e.preventDefault();
    });
    document.addEventListener('mousemove', (e) => {
        if (!isDragging) return;
        panel.style.bottom = 'auto'; panel.style.right = 'auto';
        panel.style.left = `${initialLeft + (e.clientX - startX)}px`;
        panel.style.top = `${initialTop + (e.clientY - startY)}px`;
    });
    document.addEventListener('mouseup', () => {
        if (isDragging) {
            isDragging = false;
            localStorage.setItem('antibot-panel-pos', JSON.stringify({ left: panel.style.left, top: panel.style.top }));
        }
    });

    // ==========================================
    // ⭐️高速化: 計算とテキスト処理の最適化
    // ==========================================
    // Bigram計算結果をそのまま返す（メモリ節約）
    function getBigrams(str) {
        const bigrams = new Set();
        for (let i = 0; i < str.length - 1; i++) bigrams.add(str.slice(i, i + 2));
        return bigrams;
    }

    // ⭐️高速化: 事前計算されたSet同士で比較することでループ処理を激減
    function calcSimilarityFromBigrams(bg1, bg2) {
        if (bg1.size === 0 || bg2.size === 0) return 0.0;
        let intersection = 0;
        for (let bg of bg1) { if (bg2.has(bg)) intersection++; }
        const union = bg1.size + bg2.size - intersection;
        return union === 0 ? 0 : intersection / union;
    }

    function getCoreText(str) {
        return str.replace(/[wｗ草!！?？。、・…()（）「」『』0-9a-zA-Z\s]/g, '');
    }

    // ==========================================
    // 基本情報・キャッシュ設定 (LRU対応)
    // ==========================================
    const boardDir = window.location.pathname.split('/')[1];
    const threadIdMatch = window.location.href.match(/\/res\/(\d+)/);
    const currentThreadId = threadIdMatch ? threadIdMatch[1] : "unknown";

    const cyrb53 = function(str, seed = 0) {
        let h1 = 0xdeadbeef ^ seed, h2 = 0x41c6ce57 ^ seed;
        for (let i = 0, ch; i < str.length; i++) {
            ch = str.charCodeAt(i);
            h1 = Math.imul(h1 ^ ch, 2654435761); h2 = Math.imul(h2 ^ ch, 1597334677);
        }
        h1 = Math.imul(h1 ^ (h1 >>> 16), 2246822507) ^ Math.imul(h2 ^ (h2 >>> 13), 3266489909);
        h2 = Math.imul(h2 ^ (h2 >>> 16), 2246822507) ^ Math.imul(h1 ^ (h1 >>> 13), 3266489909);
        return 4294967296 * (2097151 & h2) + (h1 >>> 0);
    };

    let cacheKeys = [];
    let globalTextDict = {};
    let whitelistDict = {};
    const CACHE_MAX = 100000;
    
    // ⭐️安定化: デバウンス処理用のフラグとタイマー
    let shouldSaveCache = false;
    let saveDebounceTimer = null;
    let statsUpdateTimer = null;

    chrome.storage.local.get('antibot-text-cache-v3', (storedDataRaw) => {
        const storedData = storedDataRaw['antibot-text-cache-v3'];
        if (storedData) {
            cacheKeys = storedData.keys || [];
            globalTextDict = storedData.dict || {};
            whitelistDict = storedData.whitelist || {};
        }

        // ⭐️安定化: 連続書き込み時のフリーズを防ぐため、遅延保存する
        function queueSaveCache() {
            if (saveDebounceTimer) clearTimeout(saveDebounceTimer);
            saveDebounceTimer = setTimeout(() => {
                if (cacheKeys.length > CACHE_MAX) {
                    const removeCount = cacheKeys.length - (CACHE_MAX - 1000);
                    const removedKeys = cacheKeys.splice(0, removeCount);
                    removedKeys.forEach(k => delete globalTextDict[k]);
                }
                chrome.storage.local.set({
                    'antibot-text-cache-v3': { keys: cacheKeys, dict: globalTextDict, whitelist: whitelistDict }
                });
                shouldSaveCache = false;
            }, 1000); // 1秒間変化がなければ保存
        }

        const postHistory = [];
        const postNoToIndex = new Map();
        const textFrequency = new Map();
        const lineFrequency = new Map();
        let isDelProcessing = false;
        let cancelDelProcess = false;

        function getThreadStats() {
            const allBlocks = Array.from(document.querySelectorAll('blockquote'));
            const totalCount = allBlocks.length;
            const botFlags = allBlocks.map(bq => bq.dataset.antibotIsBot === "true");
            const botCount = botFlags.filter(b => b).length;
            
            let maxBotRatio = 0;
            if (totalCount >= 5) {
                const windowSize = Math.min(50, totalCount);
                let currentBots = 0;
                for (let i = 0; i < windowSize; i++) if (botFlags[i]) currentBots++;
                let maxBots = currentBots;
                for (let i = windowSize; i < totalCount; i++) {
                    if (botFlags[i]) currentBots++;
                    if (botFlags[i - windowSize]) currentBots--;
                    if (currentBots > maxBots) maxBots = currentBots;
                }
                maxBotRatio = (maxBots / windowSize) * 100;
            }
            return { allBlocks, totalCount, botFlags, botCount, maxBotRatio };
        }

        function saveTextCache() {
            let cacheUpdated = false;
            let addedCount = 0;
            postHistory.forEach(post => {
                if (!post.isBot && post.normalTextJoined.length >= 6 && !post.cached) {
                    const coreText = getCoreText(post.normalTextJoined);
                    if (coreText.length >= 5) {
                        const hash = cyrb53(coreText).toString();
                        if (!globalTextDict[hash]) {
                            globalTextDict[hash] = currentThreadId; 
                            cacheKeys.push(hash); 
                            cacheUpdated = true;
                            addedCount++;
                        }
                    }
                    post.cached = true;
                }
            });
            if (cacheUpdated) queueSaveCache();
            return addedCount;
        }

        // ⭐️安定化: UIの更新もデバウンスして描画負荷を下げる
        function queueUpdateStats() {
            if (statsUpdateTimer) clearTimeout(statsUpdateTimer);
            statsUpdateTimer = setTimeout(() => {
                const stats = getThreadStats();
                const statsEl = document.getElementById('ab-stats');
                statsEl.innerText = `総:${stats.totalCount} | Bot:${stats.botCount} (最大${stats.maxBotRatio.toFixed(1)}%)`;
                
                const startBtn = document.getElementById('ab-start-btn');
                if (!isDelProcessing) {
                    if (stats.maxBotRatio >= DEL_THRESHOLD) {
                        startBtn.disabled = false; startBtn.innerText = "💣 Del送信スタート";
                    } else {
                        startBtn.disabled = true; startBtn.innerText = `Del送信 (区間${DEL_THRESHOLD}%~)`;
                    }
                }

                const miniStatsEl = document.getElementById('ab-content-mini');
                if (stats.maxBotRatio >= DEL_THRESHOLD) {
                    miniStatsEl.innerHTML = `⚠️ Bot: ${stats.maxBotRatio.toFixed(1)}%`;
                    miniStatsEl.classList.add('ab-alert');
                    miniStatsEl.title = `Botが${DEL_THRESHOLD}%を超過！クリックで展開してDel送信`;
                } else {
                    miniStatsEl.innerHTML = `Bot: ${stats.maxBotRatio.toFixed(1)}%`;
                    miniStatsEl.classList.remove('ab-alert');
                    miniStatsEl.title = "クリックでパネルを展開";
                }

                if (stats.maxBotRatio >= DEL_THRESHOLD) {
                    saveTextCache();
                }
                
                if (shouldSaveCache) queueSaveCache();
            }, 300);
        }

        document.getElementById('ab-cache-btn').addEventListener('click', () => {
            const added = saveTextCache();
            const btn = document.getElementById('ab-cache-btn');
            const originalText = btn.innerText;
            btn.innerText = `完了(+${added})`;
            btn.disabled = true;
            setTimeout(() => { btn.innerText = originalText; btn.disabled = false; }, 2500);
        });

        // ==========================================
        // 手動Del送信＆キューシステム (変更なし)
        // ==========================================
        document.getElementById('ab-start-btn').onclick = async () => {
            const startBtn = document.getElementById('ab-start-btn');
            const progressText = document.getElementById('ab-progress');
            const startNoInput = document.getElementById('ab-start-no');

            if (isDelProcessing) {
                cancelDelProcess = true; startBtn.innerText = "停止処理中..."; startBtn.disabled = true;
                return;
            }

            const stats = getThreadStats();
            const allBlocks = stats.allBlocks;
            const botFlags = stats.botFlags;
            
            const inputVal = startNoInput.value.trim();
            const minNo = inputVal ? parseInt(inputVal, 10) : 0;
            const isIndexMode = !isNaN(minNo) && minNo > 0 && minNo <= 9999;
            let delQueue = [];

            allBlocks.forEach((bq, index) => {
                let postNo = null;
                const parentContainer = bq.closest('table') || bq.parentElement;
                const cnoSpan = parentContainer.querySelector('.cno'); 
                if (cnoSpan) {
                    const match = cnoSpan.textContent.match(/No\.(\d+)/);
                    if (match) postNo = match[1];
                } 
                if (!postNo) {
                    const delcheck = parentContainer.querySelector('[id^="delcheck"]');
                    if (delcheck) postNo = delcheck.id.replace('delcheck', '');
                }
                if (!postNo) return;

                const currentPostNoNum = parseInt(postNo, 10);
                const currentPostIndex = index + 1;

                if (inputVal) {
                    if (isIndexMode) { if (currentPostIndex < minNo) return; } 
                    else { if (currentPostNoNum < minNo) return; }
                }
                if (botFlags[index]) delQueue.push(postNo);
            });

            if (delQueue.length === 0) { alert("送信対象が見つかりませんでした。"); return; }

            isDelProcessing = true; cancelDelProcess = false;
            startNoInput.disabled = true; startBtn.classList.add('stop-mode'); startBtn.innerText = "🛑 停止する";

            const totalQueue = delQueue.length;
            let processed = 0;

            while (delQueue.length > 0) {
                if (cancelDelProcess) { progressText.innerHTML = "<span style='color:#c00;'>中断しました</span>"; break; }

                const postNo = delQueue.shift();
                processed++;
                progressText.innerHTML = `送信中: No.${postNo}<br>(${processed}/${totalQueue})`;

                try {
                    const params = new URLSearchParams();
                    params.append('mode', 'post'); params.append('b', boardDir);
                    params.append('d', postNo); params.append('reason', '110'); params.append('responsemode', 'ajax');

                    const response = await fetch('/del.php', {
                        method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                        body: params.toString()
                    });
                    const resultText = await response.text();
                    if (resultText !== "ok") {
                        console.log(`[Anti-Bot] ⚠️ No.${postNo}: ${resultText}`);
                        cancelDelProcess = true;
                        progressText.innerHTML = `<span style='color:#c00;'>⚠️ エラーで緊急停止 (${resultText})</span>`;
                        break;
                    }
                } catch (err) { console.error(`[Anti-Bot] Error (No.${postNo}):`, err); }

                if (delQueue.length > 0 && !cancelDelProcess) {
                    const randomDelay = Math.floor(Math.random() * (4500 - 2500 + 1)) + 2500;
                    progressText.innerHTML = `完了: No.${postNo}<br>⏳ ${(randomDelay/1000).toFixed(1)}秒待機 (${processed}/${totalQueue})`;
                    await new Promise(resolve => setTimeout(resolve, randomDelay));
                }
            }
            isDelProcessing = false; startNoInput.disabled = false; startBtn.classList.remove('stop-mode');
            if (!cancelDelProcess) progressText.innerHTML = `<span style="color:green;">✅ 完了 (${processed}件)</span>`;
            queueUpdateStats();
        };

        // ==========================================
        // 判定・メインロジック (超高速化版)
        // ==========================================
        function processPost(blockquoteElement) {
            if (blockquoteElement.dataset.antibotChecked) return;
            blockquoteElement.dataset.antibotChecked = "true";

            const parentContainer = blockquoteElement.closest('table') || blockquoteElement.parentElement;
            const hasFile = !!parentContainer.querySelector('a[href*="src/"]');

            let currentPostNo = null;
            const cnoSpan = parentContainer.querySelector('.cno'); 
            if (cnoSpan) {
                const match = cnoSpan.textContent.match(/No\.(\d+)/);
                if (match) currentPostNo = match[1];
            } 
            if (!currentPostNo) {
                const delcheck = parentContainer.querySelector('[id^="delcheck"]');
                if (delcheck) currentPostNo = delcheck.id.replace('delcheck', '');
            }
            const currentPostIndex = postHistory.length;
            if (currentPostNo) postNoToIndex.set(currentPostNo, currentPostIndex);

            // ⭐️高速化: DOMクローンと要素作成を全廃止。HTMLの文字列操作だけで抽出する
            let htmlStr = blockquoteElement.innerHTML;
            
            // システムメッセージ(赤字)と引用(緑字)を除去
            htmlStr = htmlStr.replace(/<font[^>]*color="#ff0000"[^>]*>.*?<\/font>/gi, '');
            const hasQuote = /<font[^>]*color="#789922"[^>]*>/i.test(blockquoteElement.innerHTML);
            htmlStr = htmlStr.replace(/<font[^>]*color="#789922"[^>]*>.*?<\/font>/gi, '');
            
            const rawLines = htmlStr.split(/<br\s*\/?>/i);
            const normalLines = [];
            let normalTextJoined = "";

            for (let line of rawLines) {
                // 残ったHTMLタグを削除し、実体参照(&gt;等)を戻す
                let plainText = line.replace(/<[^>]*>?/gm, '').trim();
                plainText = plainText.replace(/&gt;/g, '>').replace(/&lt;/g, '<').replace(/&amp;/g, '&');
                
                if (plainText === "") continue;
                if (plainText.startsWith('>') || plainText.startsWith('＞')) continue; 
                
                const normalizedLine = plainText.replace(/\s+/g, '');
                if (normalizedLine === "") continue;
                if (normalizedLine.includes("ｷﾀ━━") && normalizedLine.includes("(ﾟ∀ﾟ)━━")) continue; 
                
                normalLines.push(normalizedLine);
                normalTextJoined += normalizedLine;
            }

            // ⭐️高速化: 類似度計算用Bigramを生成時のみ計算し、キャッシュとして保存
            const currentBigrams = getBigrams(normalTextJoined);
            const textContent = blockquoteElement.textContent || "";
            const normalizedText = textContent.replace(/\s+/g, '');

            const historyEntry = {
                rawText: normalizedText,
                normalTextJoined: normalTextJoined,
                bigrams: currentBigrams, // 保存しておく
                hasQuote: hasQuote,
                isBot: false,
                botReason: "",
                cached: false
            };
            postHistory.push(historyEntry);

            // ホワイトリストチェック
            const coreTextForCheck = getCoreText(normalTextJoined);
            if (coreTextForCheck.length > 0) {
                const hashForCheck = cyrb53(coreTextForCheck).toString();
                if (whitelistDict[hashForCheck]) return; 
            }

            if (currentPostIndex <= 20) {
                if (normalTextJoined.length > 0) {
                    textFrequency.set(getCoreText(normalTextJoined), (textFrequency.get(getCoreText(normalTextJoined)) || 0) + 1);
                    for (let line of normalLines) {
                        const lc = getCoreText(line);
                        if (lc.length > 0) lineFrequency.set(lc, (lineFrequency.get(lc) || 0) + 1);
                    }
                }
                return; 
            }

            let isBot = false;
            let reason = "";

            const anchorRegex = />No\.([0-9]+)/g;
            let match;
            while ((match = anchorRegex.exec(textContent)) !== null) {
                const targetNo = match[1];
                const targetElement = document.getElementById("delcheck" + targetNo);
                
                if (!targetElement) {
                    isBot = true; reason = `存在しないアンカー (>No.${targetNo})`; break;
                } else {
                    const targetIndex = postNoToIndex.get(targetNo);
                    if (targetIndex !== undefined && (currentPostIndex - targetIndex) >= 250) {
                        isBot = true; reason = `異常な遅レス (>No.${targetNo} が250レス以上前の参照)`; break;
                    }

                    const targetBlockquote = parentContainer.querySelector('blockquote');
                    if (targetBlockquote && targetBlockquote.dataset.antibotIsBot === "true") {
                        const targetReason = targetBlockquote.dataset.antibotReason || "";
                        if (!targetReason.startsWith("存在しない引用") && !targetReason.startsWith("Botへの") && !targetReason.startsWith("異常な遅レス")) {
                            const uniqueChars = new Set(normalTextJoined.split('')).size;
                            const diversity = normalTextJoined.length > 0 ? (uniqueChars / normalTextJoined.length) : 0;
                            if (diversity < 0.3) {
                                isBot = true; reason = `Botへのアンカー (>No.${targetNo})`; break;
                            }
                        }
                    }
                }
            }

            // ⭐️高速化: DOM探索の回数を最低限にする
            const quoteElements = blockquoteElement.querySelectorAll('font[color="#789922"]');
            let normalQuotes = []; 
            if (!isBot && hasQuote) {
                for (let i = 0; i < quoteElements.length; i++) {
                    const rawQuoteElementText = quoteElements[i].textContent;
                    const rawQuote = rawQuoteElementText.replace(/^>+/, '').replace(/\s+/g, '');
                    
                    const fileMatch = rawQuote.match(/^([0-9]{10,}\.(gif|jpg|png|webp|webm|mp4))$/i);
                    if (fileMatch) {
                        const fileName = fileMatch[1];
                        const fileExists = Array.from(document.querySelectorAll("a[target='_blank']")).some(link => link.textContent.trim() === fileName);
                        if (!fileExists) { isBot = true; reason = `存在しないファイルへの引用`; break; }
                        else continue;
                    }
                    
                    const idMatch = rawQuoteElementText.match(/>ID:([A-Za-z0-9\.\/\+]+)(?:\[\d+\])?/);
                    if (idMatch) {
                        const targetId = idMatch[1];
                        const idExists = Array.from(document.querySelectorAll('.cnw')).some(el => el.textContent.includes('ID:' + targetId));
                        if (!idExists) { isBot = true; reason = `存在しないIDへの引用`; break; }
                        else continue;
                    }

                    const dateMatch = rawQuoteElementText.match(/(\d{2}\/\d{2}\/\d{2}\([日月火水木金土]\)\d{2}:\d{2}:\d{2})/);
                    const noMatch = rawQuoteElementText.match(/No\.([0-9]+)/);
                    if (dateMatch || noMatch) {
                        let headerExists = false; let missingReason = "";
                        if (dateMatch) {
                            const targetDate = dateMatch[1];
                            if (Array.from(document.querySelectorAll('.cnw')).some(el => el.textContent.includes(targetDate))) headerExists = true;
                            else missingReason = `存在しない日時への引用`;
                        }
                        if (noMatch) {
                            const targetNo = noMatch[1];
                            if (document.getElementById("delcheck" + targetNo)) headerExists = true;
                            else if (!missingReason) missingReason = `存在しないレス番号への引用`;
                        }
                        if (headerExists) continue; 
                        else { isBot = true; reason = missingReason; break; }
                    }
                    
                    if (rawQuote.length >= 3) {
                        normalQuotes.push(rawQuote);
                        let foundPost = null; let foundIndex = -1;
                        for (let j = currentPostIndex - 1; j >= 0; j--) {
                            let pastPost = postHistory[j];
                            let targetTextForQuote = pastPost.isBot ? pastPost.normalTextJoined : pastPost.rawText;
                            if (targetTextForQuote.includes(rawQuote)) {
                                foundPost = pastPost; foundIndex = j; break;
                            }
                        }
                        if (!foundPost) {
                            isBot = true; reason = `存在しない引用 (>${rawQuote.substring(0, 10)}...)`; break;
                        } else {
                            if ((currentPostIndex - foundIndex) >= 250) {
                                isBot = true; reason = `異常な遅レス (引用元が250レス以上前の参照)`; break;
                            }
                            if (foundPost.isBot) {
                                const targetReason = foundPost.botReason || "";
                                if (!targetReason.startsWith("存在しない引用") && !targetReason.startsWith("Botへの") && !targetReason.startsWith("異常な遅レス")) {
                                    const uniqueChars = new Set(normalTextJoined.split('')).size;
                                    const diversity = normalTextJoined.length > 0 ? (uniqueChars / normalTextJoined.length) : 0;
                                    if (diversity < 0.3) {
                                        isBot = true; reason = `Botへの引用 (>${rawQuote.substring(0, 10)}...)`; break;
                                    }
                                }
                            }
                        }
                    }
                }

                if (!isBot && normalQuotes.length >= 2) {
                    let foundSingleSource = false;
                    for (let j = currentPostIndex - 1; j >= 0; j--) {
                        let pastPost = postHistory[j]; let containsAll = true;
                        let targetTextForQuote = pastPost.isBot ? pastPost.normalTextJoined : pastPost.rawText;
                        for (let q of normalQuotes) {
                            if (!targetTextForQuote.includes(q)) { containsAll = false; break; }
                        }
                        if (containsAll) { foundSingleSource = true; break; }
                    }
                    if (!foundSingleSource) { isBot = true; reason = "複数レスからのキメラ引用"; }
                }

                if (!isBot && normalTextJoined.length === 0 && !hasFile) {
                    isBot = true; reason = "引用のみで地の文がないレス";
                }
            }

            if (!isBot && normalTextJoined.length > 0) {
                const coreText = getCoreText(normalTextJoined);
                
                if (normalTextJoined.length >= 10) {
                    const searchLimit = Math.max(0, currentPostIndex - 100);
                    for (let i = currentPostIndex - 1; i >= searchLimit; i--) {
                        const pastPost = postHistory[i];
                        if (pastPost.normalTextJoined.length >= 10) {
                            // ⭐️高速化: 計算済みのBigram Setを使って比較。激速！
                            const sim = calcSimilarityFromBigrams(currentBigrams, pastPost.bigrams);
                            if (sim >= 0.8) {
                                isBot = true; reason = "地の文の類似コピペ (一部改変によるすり抜け)"; break;
                            }
                        }
                    }
                }

                if (!isBot && coreText.length > 0) {
                    const fullCount = (textFrequency.get(coreText) || 0) + 1;
                    textFrequency.set(coreText, fullCount);

                    if (coreText.length >= 8 && fullCount >= 2) {
                        isBot = true; reason = "地の文のコピペ再投稿 (骨格一致)";
                    } else if (coreText.length < 8 && fullCount >= SHORT_TEXT_THRESHOLD) {
                        isBot = true; reason = "地の文の短文コピペ連投 (骨格一致)";
                    }
                }

                if (!isBot) {
                    for (let line of normalLines) {
                        const lineCore = getCoreText(line);
                        if (lineCore.length === 0) continue;

                        const lineCount = (lineFrequency.get(lineCore) || 0) + 1;
                        lineFrequency.set(lineCore, lineCount);
                        
                        if (lineCore.length >= 8 && lineCount >= 2) {
                            isBot = true; reason = `地の文のコピペ (行単位: ${line.substring(0, 10)}...)`; break;
                        } else if (lineCore.length < 8 && lineCount >= SHORT_TEXT_THRESHOLD) {
                            isBot = true; reason = `地の文の短文連投 (行単位: ${line.substring(0, 10)}...)`; break;
                        }
                    }
                }
            }

            if (!isBot && normalTextJoined.length > 0) {
                const textLen = normalTextJoined.length;
                const repeatMatch = normalTextJoined.match(/(.)\1{9,}/);
                if (repeatMatch && !/[wｗー・草]/.test(repeatMatch[1])) {
                    isBot = true; reason = "ワードサラダ (無意味な文字の反復)";
                }

                if (!isBot && textLen >= 15) {
                    let kanjiCount = 0; let kanaCount = 0;
                    for (let i = 0; i < textLen; i++) {
                        const char = normalTextJoined[i];
                        if (/[\u4E00-\u9FFF\u3400-\u4DBF\uF900-\uFAFF]/.test(char)) kanjiCount++;
                        else if (/[\u30A0-\u30FF]/.test(char)) kanaCount++;
                    }
                    if (kanjiCount / textLen >= 0.8) { isBot = true; reason = "ワードサラダ (漢字率の異常)"; } 
                    else if (kanaCount / textLen >= 0.8) { isBot = true; reason = "ワードサラダ (カタカナ率の異常)"; }
                }

                if (!isBot && textLen >= 30) {
                    const uniqueChars = new Set(normalTextJoined.split('')).size;
                    if (uniqueChars / textLen < 0.15) { isBot = true; reason = "ワードサラダ (文字の多様性欠如)"; }
                }
                
                if (!isBot && textLen >= 15) {
                    const coreTextForCache = getCoreText(normalTextJoined);
                    if (coreTextForCache.length >= 10) {
                        const hash = cyrb53(coreTextForCache).toString();
                        if (globalTextDict[hash] && globalTextDict[hash] !== currentThreadId) {
                            isBot = true; reason = `他スレで記録された定型文 (一部改変も検知)`;
                            const keyIndex = cacheKeys.indexOf(hash);
                            if (keyIndex !== -1) {
                                cacheKeys.splice(keyIndex, 1); cacheKeys.push(hash);
                                shouldSaveCache = true; 
                            }
                        }
                    }
                }
            }

            if (isBot) {
                historyEntry.isBot = true; historyEntry.botReason = reason;
                blockquoteElement.dataset.antibotIsBot = "true";
                blockquoteElement.dataset.antibotReason = reason;
                
                parentContainer.classList.add('antibot-bot-post');
                const warning = document.createElement("div");
                warning.className = "antibot-warning-text"; 
                warning.style.color = "#cc4444"; warning.style.fontWeight = "bold"; warning.style.fontSize = "12px";
                warning.innerHTML = `🤔 Botの疑い: ${reason} <button class="ab-rescue-btn" style="margin-left:5px; cursor:pointer; padding:1px 4px; font-size:10px; border-radius:3px; border:1px solid #aaa; background:#fff; color:#333;">誤検知解除</button>`;
                
                warning.querySelector('.ab-rescue-btn').addEventListener('click', () => {
                    blockquoteElement.dataset.antibotIsBot = "false";
                    parentContainer.classList.remove('antibot-bot-post');
                    warning.remove(); historyEntry.isBot = false;
                    if (normalTextJoined.length > 0) {
                        const hash = cyrb53(getCoreText(normalTextJoined)).toString();
                        whitelistDict[hash] = true; queueSaveCache();
                    }
                    queueUpdateStats();
                });

                blockquoteElement.parentNode.insertBefore(warning, blockquoteElement);
            }
        }

        function scanExistingPosts() {
            const blockquotes = document.querySelectorAll('blockquote');
            blockquotes.forEach(bq => processPost(bq));
            queueUpdateStats();
        }

        function observeNewPosts() {
            const observer = new MutationObserver((mutations) => {
                let hasNewPost = false;
                mutations.forEach((mutation) => {
                    mutation.addedNodes.forEach((node) => {
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            if (node.tagName && node.tagName.toLowerCase() === 'blockquote') {
                                processPost(node); hasNewPost = true;
                            } else {
                                const blockquotes = node.querySelectorAll('blockquote');
                                if (blockquotes.length > 0) {
                                    blockquotes.forEach(bq => processPost(bq)); hasNewPost = true;
                                }
                            }
                        }
                    });
                });
                if (hasNewPost) {
                    queueUpdateStats(); // ⭐️高速化: 更新を一括処理
                }
            });
            observer.observe(document.body, { childList: true, subtree: true });
        }

        scanExistingPosts();
        observeNewPosts();
    });
}