import json
import math
import os
import sys
import tkinter as tk
import tkinter.filedialog as tkFileDialog
import tkinter.messagebox
from dataclasses import dataclass
from enum import Enum
from pydantic import BaseModel, Field
from tkinter import ttk
from tkinter.font import Font as tkFont
from tkinter.font import nametofont
from typing import ClassVar

import piexif
from PIL import Image, ImageDraw, ImageFilter, ImageTk
from PIL.PngImagePlugin import PngInfo
import pillow_avif
import struct

# 初期値を変更したい場合は下記の数値を変更してください。
DEFAULT_MOSAIC_SIZE = 10  # モザイクサイズ(初期値: 10)
DEFAULT_PEN_SIZE = 30  # 塗りつぶしペンサイズ(初期値: 30)
DEFAULT_FILTER_MODE = 0  # フィルタモード(0:モザイク, 1:ぼかし, 2:黒塗り)(初期値: 0)
DEFAULT_SELECTION_MODE = 0  # 選択モード(0:矩形, 1:ペン)(初期値: 0)
DEFAULT_TOOLBAR_FONT_SIZE = 12  # ツールバーフォントサイズ(初期値: 12)
# モザイクサイズボタンの初期値と表示名
# 初期値: ((10, '小'), (15, '中'), (20, '大'))
DEFAULT_MOSAIC_SIZE_BUTTON = ((10, '小'), (15, '中'), (20, '大'))
# 色の設定(#rrggbb形式または'white','red'などの色名)
DEFAULT_PEN_CURSOR_COLOR = '#00ff00'  # ペンカーソルの色(初期値(緑): '#00ff00')
DEFAULT_PEN_DRAW_COLOR = '#00ff00'  # ペン塗りつぶしの色(初期値(緑): '#00ff00')

MOSAIC_SIZE_MIN = 10  # モザイクサイズ最小値
MOSAIC_SIZE_MAX = 50  # モザイクサイズ最大値
PEN_SIZE_MIN = 10  # 塗りつぶしペンサイズ最小値
PEN_SIZE_MAX = 50  # 塗りつぶしペンサイズ最大値
FILTER_MODE_MIN = 0  # フィルタモード最小値
FILTER_MODE_MAX = 2  # フィルタモード最大値
SELECTION_MODE_MIN = 0  # 選択モード最小値
SELECTION_MODE_MAX = 1  # 選択モード最大値

SCALE_MIN = 20  # 表示倍率最小値(%)
SCALE_MAX = 400  # 表示倍率最大値(%)
SCALE_DEFAULT = 100  # デフォルト表示倍率(%)
SCALE_STEP = 20  # 表示倍率ステップ値(%)

WINDOW_GEOMETRY_FILE = 'window_geometry.json'
DRAWING_SETTINGS_FILE = 'drawing_settings.json'

class DrawinfSettings(BaseModel):
    mosaic_size: int = Field(default=DEFAULT_MOSAIC_SIZE, ge=MOSAIC_SIZE_MIN, le=MOSAIC_SIZE_MAX)
    pen_size: int = Field(default=DEFAULT_PEN_SIZE, ge=PEN_SIZE_MIN, le=PEN_SIZE_MAX)
    filter_mode: int = Field(default=DEFAULT_FILTER_MODE, ge=FILTER_MODE_MIN, le=FILTER_MODE_MAX)
    selection_mode: int = Field(default=DEFAULT_SELECTION_MODE, ge=SELECTION_MODE_MIN, le=SELECTION_MODE_MAX)

# 画像領域選択モード
class SelectionMode(Enum):
    # 矩形選択
    RECT = 0
    # ペン塗りつぶし選択
    PEN = 1


# ペン管理クラス(何もボタンを押していないときにマウスに追尾する丸いペンのカーソルと描画の表示を管理する)
class PenManager:
    def __init__(self, canvas: tk.Canvas, pen_size: int, main_color: str, cursor_style: dict[str, any] = None):
        # ペンの描画色
        self.color = main_color
        # ペンのサイズ
        self.pen_size = pen_size
        # ペンのカーソルスタイル
        self.cursor_style = cursor_style or {}
        self.change_canvas(canvas)
        # 表示倍率（キャンバス上の表示に使う）
        self.scale = SCALE_DEFAULT

    def change_canvas(self, canvas: tk.Canvas) -> None:
        self.canvas = canvas

        # カーソルの表示状態
        self.cursor_visible = False
        # 描画中の線分の表示状態
        self.drawing_visible = False

        # 描画中の線分の座標[x0, y0, x1, y1, x2, y2, ...]
        # line_coords は表示座標（キャンバス座標）で保持する
        self.line_coords: list[float] = []
        # カーソルの座標
        self.cursor_pos = (0.0, 0.0)

        # 最初の点（ovalでは点々塗りつぶしができないのでpolygonでごまかす）
        # self.start_point = canvas.create_oval(
        #     0, 0, self.pen_size, self.pen_size, fill=self.color, width=0, stipple='gray25', state=tk.HIDDEN
        # )
        self.start_point = canvas.create_polygon(
            0, 0, self.pen_size, 0, self.pen_size, self.pen_size, 0, self.pen_size, fill=self.color, width=0, stipple='gray25', smooth=True, splinesteps=64, joinstyle=tk.MITER, state=tk.HIDDEN
        )
        # 描画中の線分
        self.lines = canvas.create_line(
            0, 0, self.pen_size, self.pen_size, fill=self.color, width=self.pen_size, stipple='gray25', state=tk.HIDDEN
        )
        # ペンのカーソル（ovalでは点々塗りつぶしができないのでpolygonでごまかす）
        self.pen_cursor = canvas.create_polygon(
            0, 0, self.pen_size, 0, self.pen_size, self.pen_size, 0, self.pen_size, stipple='gray25', smooth=True, splinesteps=64, joinstyle=tk.MITER, state=tk.HIDDEN, **self.cursor_style
        )

    def set_pen_size(self, size: int) -> None:
        self.pen_size = size
        if self.cursor_visible:
            self.move_cursor(*self.cursor_pos)

    def set_scale(self, scale: int) -> None:
        """表示スケールを更新する。ペン表示のサイズをスケールに合わせて再描画する"""
        self.scale = scale
        # 既存のカーソル表示を現在の位置で再描画して大きさを更新
        if self.cursor_visible:
            self.move_cursor(*self.cursor_pos)
        # 線の太さをスケールに合わせる
        disp_width = max(1, int(self.pen_size * max(0.01, self.scale / 100)))
        try:
            self.canvas.itemconfigure(self.lines, width=disp_width)
        except Exception:
            pass

    def move_cursor(self, x: float, y: float) -> None:
        # pen_size は画像上のピクセル単位なので、表示上の大きさは scale をかける
        disp_pen = max(1, int(self.pen_size * max(0.01, self.scale / 100)))
        hp = disp_pen // 2
        self.canvas.coords(self.pen_cursor, x - hp, y - hp, x + hp, y - hp, x + hp, y + hp, x - hp, y + hp)
        self.cursor_pos = (x, y)

    def set_cursor_visible(self, visible: bool) -> None:
        if self.cursor_visible == visible:
            return
        self.cursor_visible = visible
        self.canvas.itemconfigure(self.pen_cursor, state=tk.NORMAL if visible else tk.HIDDEN)

    def start_drawing(self, x: float, y: float) -> None:
        # x,y は表示座標（キャンバス座標）で渡されることを想定
        self.line_coords = [x, y]
        disp_pen = max(1, int(self.pen_size * max(0.01, self.scale / 100)))
        hp = disp_pen // 2
        self.canvas.coords(self.start_point, x - hp, y - hp, x + hp, y - hp, x + hp, y + hp, x - hp, y + hp)
        self.canvas.itemconfigure(self.start_point, state=tk.NORMAL)
        self.canvas.tag_lower(self.start_point, self.pen_cursor)
        self.drawing_visible = True

    def move_pen(self, x: float, y: float) -> None:
        if x == self.line_coords[-2] and y == self.line_coords[-1]:
            # 最後に移動した座標と同じ場合は無視する
            return

        # ついでにカーソルも移動させる
        self.move_cursor(x, y)

        self.line_coords.extend([x, y])
        self.canvas.coords(self.lines, *self.line_coords)

        if len(self.line_coords) == 4:
            # 最初の線分描画時にカーソルを線分より前面に移動する
            disp_width = max(1, int(self.pen_size * max(0.01, self.scale / 100)))
            self.canvas.itemconfigure(self.lines, state=tk.NORMAL, width=disp_width)
            self.canvas.tag_lower(self.lines, self.pen_cursor)

    def drawing_finished(self) -> list[float] | None:
        self.canvas.itemconfigure(self.lines, state=tk.HIDDEN)
        self.canvas.itemconfigure(self.start_point, state=tk.HIDDEN)
        self.drawing_visible = False
        if len(self.line_coords) <= 2:
            # 1点=最初の点からマウスを動かしていない場合は無視
            return None
        # line_coords は表示座標なので、実画像上の座標に変換して返す
        if self.scale and self.scale != 100:
            return [c / (self.scale / 100) for c in self.line_coords]
        return list(self.line_coords)


#Undo情報を保有するクラス
@dataclass
class Undo:
    # 画像全域を示す矩形の座標
    WHOLE_IMAGE_AREA: ClassVar[tuple[int, int, int, int]] = (-1, -1, -1, -1)

    region: Image
    rect: tuple[int, int, int, int] = WHOLE_IMAGE_AREA


class App(tk.Frame):

    def __init__(self, master=None):
        super().__init__(master)
        self.master = master
        self.load_window_geometry()
        self.load_drawing_settings()

        # 表示倍率(%)
        self.scale: int = SCALE_DEFAULT

        # 処理モード('切抜','縮小'についてはペン使用時は動作しない->'モザイク'を選択時と同様の動作をする)
        # self.applist = ['モザイク','ぼかし','黒塗り','切抜','縮小']
        self.applist = ['モザイク', 'ぼかし', '黒塗り']
        self.mode = self.initial_draw_settings.filter_mode
        self.filename = ''

        # 画像ファイルリストの配列変数
        self.image_file_list = []
        self.image_file_index = 0

        if (len(sys.argv[1:])==1):
            arguments = sys.argv[1]
            print(arguments)

            #引数がフォルダの時
            if os.path.isdir(arguments):
                #フォルダ内の画像ファイルを名前順でソートし画像ファイルリストにすべて追加する
                folder_path = arguments
                for file_name in sorted(os.listdir(folder_path)):
                    if file_name.endswith('.jpg') or file_name.endswith('.jpeg') or file_name.endswith('.png') or file_name.endswith('.gif') or file_name.endswith('.webp') or file_name.endswith('.avif'):
                        self.image_file_list.append(os.path.join(folder_path, file_name))

                #画像ファイルリストにデータが存在する時
                if (len(self.image_file_list) > self.image_file_index):
                    # 画像ファイルリストから先頭のファイルを取得する
                    image_file = self.image_file_list[self.image_file_index]
                    self.load_image(image_file)
                #画像ファイルリストにデータが存在しない時
                else:
                    print('引数で指定されたフォルダ内に、画像ファイルが見つかりませんでした。')
                    tkinter.messagebox.showinfo("終了", "引数で指定されたフォルダ内に、画像ファイルが見つかりませんでした。")
                    sys.exit(1)

            #引数がファイルの時
            else:
                self.load_image(arguments)

        else:
            self.load_image()

        self.create_toolbar(
            parent = self,
            init_mosaic_size = self.initial_draw_settings.mosaic_size,
            init_pen_size = self.initial_draw_settings.pen_size,
            init_selection_mode = self.initial_draw_settings.selection_mode,
            init_filter_mode = self.initial_draw_settings.filter_mode)

        def on_pen_size_change(*_):
            if pm := getattr(self, 'pen_manager', None):
                pm.set_pen_size(self.pen_size_var.get())

        self.pen_size_var.trace_add("write", on_pen_size_change)

        self.create_canvas()

        self.pack()

        # キーボードイベントを登録する
        self.master.bind_all("<Control-z>", self.on_undo)
        self.master.bind_all("<Control-n>", self.on_next)
        self.master.bind_all("<Right>", self.on_next)
        self.master.bind_all("<Control-b>", self.on_back)
        self.master.bind_all("<Left>", self.on_back)

        # ウィンドウサイズ変更イベント <Configure> を resize_window 関数にバインド
        self.master.bind("<Configure>", self.resize_window)

        # ズーム用キー（Ctrl+0でリセット）
        self.master.bind_all("<Control-0>", lambda e: self.reset_zoom())
        # Ctrl + +/- 系のショートカットを処理するハンドラを登録
        self.master.bind_all("<Control-KeyPress>", self.on_ctrl_keypress)

        # 右クリックメニュー作成
        idx = 0
        self.popup_menu = tk.Menu(self, tearoff=0)
        self.popup_menu.add_command(label="全て元に戻す", state=tk.DISABLED, command=self.on_undo_all)
        self.undo_all_menu_index = idx
        idx += 1
        self.popup_menu.add_command(label="保存せずに終了する", command=self.on_close_window_without_save_image)

    def resize_window(self, event):
        # ウィンドウサイズに合わせてselfのサイズを更新する。※内部に配置されているフレームやカンバス、スクロールバーはpackのfill指定で自動的に連鎖してリサイズされる
        self.config(width=self.master.winfo_width(), height=self.master.winfo_height())

    #画像系のオブジェクトの開放
    def del_image(self):
        del self.write_path
        self.image.close()
        del self.image
        del self.parameters
        del self.prompt
        del self.workflow
        del self.image_width
        del self.image_height
        self.image_tk.__del__()
        del self.image_tk

    #キャンバス系のオブジェクトの開放
    def del_canvas(self):
        # フレームを削除する
        self.frame.pack_forget()
        self.frame = None

        # スクロールバーを削除する
        self.canvas.config(yscrollcommand=None)
        self.canvas.config(xscrollcommand=None)
        self.scrollbar_vertical.pack_forget()
        self.scrollbar_horizontal.pack_forget()
        self.scrollbar_vertical.destroy()
        self.scrollbar_horizontal.destroy()
        self.scrollbar_vertical = None
        self.scrollbar_horizontal = None

        # 画像を削除する
        self.canvas.delete(self.image_on_canvas)
        self.image_on_canvas = None

        # マウスイベントを解除する
        self.canvas.unbind("<ButtonPress-1>")
        self.canvas.unbind("<B1-Motion>")
        self.canvas.unbind("<Motion>")
        self.canvas.unbind("<ButtonRelease-1>")
        self.canvas.unbind_all("<MouseWheel>")
        self.canvas.unbind_all("<Shift-MouseWheel>")
        self.canvas.unbind_all("<ButtonPress-3>")
        #  マウスドラッグによるスクロール用のイベントを解除
        self.canvas.unbind("<ButtonPress-2>")
        self.canvas.unbind("<B2-Motion>")
        self.canvas.unbind("<ButtonRelease-2>")


        # 矩形描画用の変数を初期化する
        self.start_x = None
        self.start_y = None
        self.end_x = None
        self.end_y = None
        self.rect = None
        self.pen_manager = None

        # Undo用の配列変数を初期化する
        self.clear_undo_list()

        # キャンバスを削除する
        self.canvas.destroy()
        self.canvas = None

    def create_toolbar(self, parent, init_mosaic_size=DEFAULT_MOSAIC_SIZE, init_pen_size=DEFAULT_PEN_SIZE, init_selection_mode=DEFAULT_SELECTION_MODE, init_filter_mode=DEFAULT_FILTER_MODE):
        # ウィジェット変数の作成
        self.selection_mode_var = tk.IntVar(value=init_selection_mode)
        self.mosaic_size_var = tk.IntVar()
        self.filter_mode_var = tk.IntVar(value=init_filter_mode)
        self.pen_size_var = tk.IntVar()
        self.pen_size_label_var = tk.StringVar()
        self.pen_size_var.trace_add("write", lambda *_: self.pen_size_label_var.set(f"{self.pen_size_var.get()}\npx"))
        self.mosaic_size_label_var = tk.StringVar()
        self.mosaic_size_var.trace_add(
            "write", lambda *_: self.mosaic_size_label_var.set(f"{self.mosaic_size_var.get()}px")
        )

        def pack_sep(toolbar, vertical: bool = True):
            """ツールバーにセパレータを追加する"""
            sep = ttk.Separator(toolbar, orient='vertical' if vertical else 'horizontal')
            sep.pack(side=tk.LEFT, fill='y' if vertical else 'x', padx=2, pady=2)

        def set_takefocus(widget, value=0):
            """子ウィジェットのフォーカスを無効にする"""
            for child in widget.winfo_children():
                child.configure(takefocus=0)

        # ツールバー用フォントの作成(デフォルトフォントのコピーを作りサイズを変更する)
        toolbar_font = tkFont(font="TkDefaultFont")
        toolbar_font.configure(size=DEFAULT_TOOLBAR_FONT_SIZE, weight='bold')

        default_font = nametofont("TkDefaultFont")

        # 上部ツールバーと左側ツールバーに乗っているウィジェットのデフォルトフォントを設定する
        parent.option_add('*topToolbar*Font', toolbar_font)
        parent.option_add('*leftToolbar*Font', toolbar_font)

        # 上部ツールバー
        toolbar = tk.Frame(parent, name='topToolbar', takefocus=0, bd=1, relief=tk.RAISED)

        btn = tk.Button(toolbar, text="←", command=lambda: self.on_back(None))
        btn.pack(side=tk.LEFT, padx=2, pady=2)
        btn = tk.Button(toolbar, text="→", command=lambda: self.on_next(None))
        btn.pack(side=tk.LEFT, padx=2, pady=2)

        pack_sep(toolbar)

        self.undo_btn = tk.Button(toolbar, text="元に戻す", state=tk.DISABLED, command=lambda: self.on_undo(None))
        self.undo_btn.pack(side=tk.LEFT, padx=2, pady=2)

        pack_sep(toolbar)

        lbl = tk.Label(toolbar, text="モード")
        lbl.pack(side=tk.LEFT, padx=2, pady=2)

        for i, app in enumerate(self.applist):
            btn = tk.Radiobutton(
                toolbar,
                text=app,
                indicatoron=0,
                command=self.on_filter_toolbar_click,
                variable=self.filter_mode_var,
                value=i,
            )
            btn.pack(side=tk.LEFT, padx=2, pady=2)

        pack_sep(toolbar)

        lbl = tk.Label(toolbar, text="モザイク")
        lbl.pack(side=tk.LEFT, padx=2, pady=2)

        lbl = tk.Label(toolbar, textvariable=self.mosaic_size_label_var, justify=tk.RIGHT, font=default_font, width=4)
        lbl.pack(side=tk.LEFT, padx=2, fill='y')

        # モザイクサイズごとのラジオボタンを作成する
        for sz, name in DEFAULT_MOSAIC_SIZE_BUTTON:
            btn = tk.Radiobutton(toolbar, text=name, indicatoron=0, variable=self.mosaic_size_var, value=sz)
            btn.pack(side=tk.LEFT, padx=2, pady=2)

        self.guideline_mosaic_button = tk.Radiobutton(
            toolbar, text="渋", indicatoron=0, variable=self.mosaic_size_var, value=0
        )
        self.guideline_mosaic_button.pack(side=tk.LEFT, padx=2, pady=2)

        scale_mosaic = tk.Scale(
            toolbar,
            orient=tk.HORIZONTAL,
            from_=MOSAIC_SIZE_MIN,
            to=MOSAIC_SIZE_MAX,
            length=150,
            variable=self.mosaic_size_var,
            showvalue=0,
        )
        scale_mosaic.pack(side=tk.LEFT, padx=2, pady=2)
        scale_mosaic.bind_all("<Control-Shift-MouseWheel>", self.on_ctrl_shift_mouse_wheel)

        pack_sep(toolbar)

        btn_mask = tk.Button(toolbar, text="マスク画像読込", command=lambda: self.on_mask(None))
        btn_mask.pack(side=tk.LEFT, padx=2, pady=2)
        btn_mask.bind_all("<m>", lambda e: btn_mask.invoke())

        pack_sep(toolbar)

        btn_restore_drawing_settings = tk.Button(toolbar, text="デフォルト描画設定", command=lambda: self.on_restore_draw_settings(None))
        btn_restore_drawing_settings.pack(side=tk.LEFT, padx=2, pady=2)
        btn_restore_drawing_settings.bind_all("<d>", lambda e: btn_restore_drawing_settings.invoke())

        btn_save_drawing_settings = tk.Button(toolbar, text="描画設定保存", command=lambda: self.save_drawing_settings())
        btn_save_drawing_settings.pack(side=tk.LEFT, padx=2, pady=2)
        btn_save_drawing_settings.bind_all("<s>", lambda e: btn_save_drawing_settings.invoke())

        toolbar.pack(side=tk.TOP, fill=tk.X)

        # ツールバー上のウィジェットのフォーカスを無効にする
        set_takefocus(toolbar, 0)

        # 左側ツールバー

        def tategaki(text, shortcut=None):
            return '\n'.join(list(text)) + (f'\n{shortcut}' if shortcut else '')

        left_toolbar = tk.Frame(parent, name='leftToolbar', takefocus=0, bd=1, relief=tk.RAISED)

        lbl = tk.Label(left_toolbar, text=tategaki("選択"))
        lbl.pack(side=tk.TOP, padx=2, pady=2)

        btn_rect = tk.Radiobutton(
            left_toolbar,
            text=tategaki("矩形"),
            variable=self.selection_mode_var,
            indicatoron=0,
            value=SelectionMode.RECT.value,
            command=self.on_selection_mode_change,
        )
        btn_rect.pack(side=tk.TOP, padx=2, pady=2)
        btn_rect.bind_all("<r>", lambda e: btn_rect.invoke())

        btn_pen = tk.Radiobutton(
            left_toolbar,
            text=tategaki("ペン"),
            variable=self.selection_mode_var,
            indicatoron=0,
            value=SelectionMode.PEN.value,
            command=self.on_selection_mode_change,
        )
        btn_pen.pack(side=tk.TOP, padx=2, pady=2)
        btn_pen.bind_all("<p>", lambda e: btn_pen.invoke())

        # label = tk.Label(left_toolbar, text=tategaki('ペンサイズ'), font=toolbar_font)
        # label.pack(side=tk.TOP, padx=2, pady=2)
        lbl = tk.Label(left_toolbar, textvariable=self.pen_size_label_var, font=default_font)
        lbl.pack(side=tk.TOP, padx=2, pady=2)

        scale_pen = tk.Scale(
            left_toolbar,
            orient=tk.VERTICAL,
            from_=PEN_SIZE_MIN,
            to=PEN_SIZE_MAX,
            length=150,
            resolution=5,
            variable=self.pen_size_var,
            command=lambda x: self.move_pen_cursor_to_center(),
            showvalue=0,
        )
        scale_pen.pack(side=tk.TOP, padx=2, pady=2)
        scale_pen.bind_all("<Control-MouseWheel>", self.on_ctrl_mouse_wheel)

        left_toolbar.pack(side=tk.LEFT, fill=tk.Y)

        # 左ツールバー上のウィジェットのフォーカスを無効にする
        set_takefocus(left_toolbar, 0)

        self.mosaic_size_var.set(init_mosaic_size)
        self.pen_size_var.set(init_pen_size)
        # ズーム表示パーセントラベル
        self.zoom_label_var = tk.StringVar(value=f"{int(self.scale)}%")

        # ズームボタンをツールバーに追加（右側に）
        pack_sep(toolbar)
        btn_zoom_out = tk.Button(toolbar, text="-", width=3, command=lambda: self.zoom_out())
        btn_zoom_out.pack(side=tk.RIGHT, padx=2, pady=2)
        lbl_zoom = tk.Label(toolbar, textvariable=self.zoom_label_var)
        lbl_zoom.pack(side=tk.RIGHT, padx=2, pady=2)
        btn_zoom_in = tk.Button(toolbar, text="+", width=3, command=lambda: self.zoom_in())
        btn_zoom_in.pack(side=tk.RIGHT, padx=2, pady=2)

    def create_canvas(self):
        # キャンバスを作成する
        self.frame = tk.Frame(self)
        self.frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)

        self.canvas = tk.Canvas(self.frame)
        self.canvas.pack(side=tk.TOP, fill=tk.BOTH, expand=True)
        self.canvas.focus_force()
        # 表示サイズに合わせたスクロール領域を設定
        self.canvas.configure(scrollregion=(0, 0, int(self.image_width * self.scale / 100), int(self.image_height * self.scale / 100)))
        self.canvas.xview_moveto(0)
        self.canvas.yview_moveto(0)

        # スクロールバーを作成する
        self.scrollbar_vertical = tk.Scrollbar(self, orient=tk.VERTICAL)
        self.scrollbar_vertical.pack(side=tk.RIGHT, fill=tk.Y)
        self.scrollbar_vertical.config(command=self.canvas.yview)
        self.canvas.config(yscrollcommand=self.scrollbar_vertical.set)
        # 横方向のスクロールバーを作成する
        self.scrollbar_horizontal = tk.Scrollbar(self.frame, orient=tk.HORIZONTAL)
        self.scrollbar_horizontal.pack(side=tk.BOTTOM, fill=tk.X)
        self.scrollbar_horizontal.config(command=self.canvas.xview)
        self.canvas.config(xscrollcommand=self.scrollbar_horizontal.set)
        # 画像をキャンバスに表示する（表示用にスケールされた PhotoImage を使用）
        self.image_on_canvas = self.canvas.create_image(0, 0, anchor=tk.NW, image=self.image_tk)

        # マウスイベントを登録する
        self.canvas.bind("<ButtonPress-1>", self.on_button_press)
        self.canvas.bind("<B1-Motion>", self.on_move_press)
        self.canvas.bind("<Motion>", self.on_mouse_move)
        self.canvas.bind("<ButtonRelease-1>", self.on_button_release)
        self.canvas.bind_all("<MouseWheel>", self.on_mouse_wheel)
        self.canvas.bind_all("<Shift-MouseWheel>", self.on_shift_mouse_wheel)
        self.canvas.bind_all("<ButtonPress-3>", self.on_right_button_press)
        # Alt + マウスホイール でズーム
        self.canvas.bind_all("<Alt-MouseWheel>", self.on_alt_mouse_wheel)
        #  マウスドラッグによるスクロール用のイベントを登録する
        self.canvas.bind("<ButtonPress-2>", lambda e: [self.canvas.scan_mark(e.x, e.y), root.configure(cursor="fleur")])
        self.canvas.bind("<B2-Motion>", lambda e: self.canvas.scan_dragto(e.x, e.y, gain=1))
        self.canvas.bind("<ButtonRelease-2>", lambda e: root.configure(cursor=""))

        # 矩形を描画するための変数
        self.start_x = None
        self.start_y = None
        self.end_x = None
        self.end_y = None
        self.rect = None

        # Undo用の配列変数
        self.clear_undo_list()

        self.pen_manager = PenManager(
            self.canvas,
            self.pen_size_var.get(),
            DEFAULT_PEN_DRAW_COLOR,
            cursor_style={'fill': DEFAULT_PEN_CURSOR_COLOR, "outline": DEFAULT_PEN_CURSOR_COLOR, 'width': 1},
        )
        self.pen_manager.set_scale(SCALE_DEFAULT)
        self.calc_guideline_mosaic_size()

    def load_image(self, file=None):

        if file is None:
            # ファイルを選択する
            filepath = tkFileDialog.askopenfilename(filetypes=[("画像ファイル", "*.jpg;*.jpeg;*.png;*.gif;*.webp;*.avif")])
            # ファイル選択がない時、強制終了する
            if not filepath: sys.exit(1)
        else:
            filepath = file

        # ファイル名、拡張子、フルパスを取得する
        basename = os.path.basename(filepath)
        name, ext = os.path.splitext(basename)
        dirname = os.path.dirname(filepath)

        #フォルダ指定モードの時（ファイルリストが存在する）時
        if (len(self.image_file_list) > 0):
            # フルパスを組み立てる（フォルダ名に "_mc" を追加する。ファイル名はそのまま。）
            new_filepath = os.path.join(dirname + "_mc", name + ext)
        else:
            # フルパスを組み立てる（ファイル名に "_mc" を追加する）
            new_filepath = os.path.join(dirname, name + "_mc" + ext)

        self.write_path=new_filepath
        self.image = Image.open(filepath)
        self.lossless = self.check_lossless(filepath)
        image_format = self.image.format.lower() if self.image.format else 'unknown'
        self.filename = basename
        root.title(self.filename)
        try:
            parameters =self.image.text['parameters']
        except:
            parameters =''
        self.parameters=parameters
        try:
            prompt =self.image.text['prompt']
        except:
            prompt =''
        self.prompt=prompt
        try:
            workflow =self.image.text['workflow']
        except:
            workflow =''
        self.workflow=workflow
        self.image_width, self.image_height = self.image.size
        # 表示倍率(%)
        self.set_scale(SCALE_DEFAULT)

    def check_lossless(self, filepath):
        image_format = self.image.format.lower() if self.image.format else 'unknown'
        # jpeg,avifの判定は非サポート
        if (image_format == 'png'):
            return True
        if (image_format == 'webp'):
            # pillow-avif-plugin 1.5.2ではWebPのロスレスチェックができないため
            # WebP内から'VP8L'を探しロスレスかどうか判断する
            # https://developers.google.com/speed/webp/docs/riff_container
            with open(filepath, 'br') as f:
                header = f.read(12)
                riff, data_lenth, webp = struct.unpack('<4sI4s', header)
                if (riff != b'RIFF' or webp != b'WEBP'):
                    return False
                data_lenth -= len(webp)  # 'WEBP'の長さを引く

                while data_lenth > 0:
                    header = f.read(8)
                    chunk_fourCC, chunk_size = struct.unpack('<4sI', header)
                    f.seek(chunk_size, 1)
                    data_lenth -= chunk_size + len(header)
                    if chunk_fourCC == b'VP8L':
                        return True     # WebP(Lossless)
                    if chunk_fourCC == b'VP8 ':
                        return False    # WebP(Lossy)
        return False

    def move_pen_cursor_to_center(self):
        cs = self.canvas
        self.pen_manager.move_cursor(cs.canvasx(cs.winfo_width() / 2), cs.canvasy(cs.winfo_height() / 2))

    def clear_undo_list(self):
        self.undo_list = []

        if hasattr(self, 'undo_btn'):
            self.undo_btn.configure(state=tk.DISABLED)
        if hasattr(self, 'popup_menu'):
            self.popup_menu.entryconfigure(self.undo_all_menu_index, state=tk.DISABLED)

    def add_undo_list(self, crop_rect: tuple[int, int, int, int] = Undo.WHOLE_IMAGE_AREA):
        """
        self.imageをUndoリストに追加する

        :param crop_rect: 切り抜き範囲(省略時は切り抜きを行わず、画像全域を使用)
        """
        if crop_rect != Undo.WHOLE_IMAGE_AREA:
            image = self.image.crop(crop_rect)
        else:
            image = self.image.copy()

        self.undo_list.append(Undo(image, crop_rect))

        self.undo_btn.configure(state=tk.NORMAL)
        self.popup_menu.entryconfigure(self.undo_all_menu_index, state=tk.NORMAL)

    def on_undo(self, event):
        #Undoリストに復元用データが存在する時
        if (len(self.undo_list) > 0):
            # 範囲をUndoリストから復元する
            undo = self.undo_list.pop()

            #切り抜きと縮小のUndo処理の時（すべての座標が-1）
            if undo.rect == Undo.WHOLE_IMAGE_AREA:
                # 画像（全体）を更新する
                self.image = undo.region
            else:
                # 画像（差分）を更新する
                self.image.paste(undo.region, undo.rect)

            # キャンバスに表示する画像を更新する
            self.image_tk = ImageTk.PhotoImage(self.image)
            self.update_display_image()

        if len(self.undo_list) == 0:
            self.undo_btn.configure(state=tk.DISABLED)
            self.popup_menu.entryconfigure(self.undo_all_menu_index, state=tk.DISABLED)

    def on_undo_all(self):
        """全てのUndoを実行する"""
        while len(self.undo_list) > 0:
            self.on_undo(None)

    #指定されたフォルダパスが存在しない場合は、再帰的にフォルダを作成する処理
    def create_folder_if_not_exist(self, path):
        if not os.path.exists(path):
            os.makedirs(path)

    #次の画像ファイルを開く処理（フォルダ指定モード用の処理）
    def on_next(self, event):
        #フォルダ指定モードではない（ファイルリストが存在しない）時、何も処理せず終了する
        if (len(self.image_file_list) == 0): return

        #画像イメージのインデックスを加算する
        self.image_file_index += 1
        #画像ファイルリストのインデックスが有効な時
        if (len(self.image_file_list) > self.image_file_index):
            #画像をセーブする
            self.save_image()

            # 画像ファイルリストから該当インデックスの画像ファイルを取得する
            image_file = self.image_file_list[self.image_file_index]

            self.del_image()
            self.del_canvas()
            self.pack_forget()

            self.load_image(image_file)
            self.create_canvas()
            self.pack()

        #画像ファイルリストのインデックスが無効な時
        else:
            result = tkinter.messagebox.askyesno("終了", "現在開いている画像はフォルダ内の最後の画像です。次の画像ファイルはありません。終了しますか？")
            if result:
                #画像をセーブする
                self.save_image()
                self.save_window_geometry()
                root.destroy()
            #画像イメージのインデックスを戻す
            self.image_file_index -= 1

    #前の画像ファイルを開く処理（フォルダ指定モード用の処理）
    def on_back(self, event):
        #フォルダ指定モードではない（ファイルリストが存在しない）時、何も処理せず終了する
        if (len(self.image_file_list) == 0): return

        #画像イメージのインデックスを減算する
        self.image_file_index -= 1
        #画像ファイルリストのインデックスが有効な時
        if (0 <= self.image_file_index):
            #画像をセーブする
            self.save_image()

            # 画像ファイルリストから該当インデックスの画像ファイルを取得する
            image_file = self.image_file_list[self.image_file_index]

            self.del_image()
            self.del_canvas()
            self.pack_forget()

            self.load_image(image_file)
            self.create_canvas()
            self.pack()

        #画像ファイルリストのインデックスが無効な時
        else:
            result = tkinter.messagebox.askyesno("終了", "現在開いている画像はフォルダ内の先頭の画像です。前の画像ファイルはありません。終了しますか？")
            if result:
                #画像をセーブする
                self.save_image()
                self.save_window_geometry()
                root.destroy()
            #画像イメージのインデックスを戻す
            self.image_file_index += 1

    def on_mask(self, event):
        mask_files = tkFileDialog.askopenfilenames(filetypes=[("画像ファイル", "*.jpg;*.jpeg;*.png;*.gif;*.webp;*.avif")])
        # ファイル選択がない時は何もしない
        if not mask_files:
            return

        masks = []
        for file in mask_files:
            try:
                masks.append(Image.open(file).convert("L"))
            except Exception as e:
                print(f"マスク画像の読込に失敗しました: {file}")
                tkinter.messagebox.showinfo(title="エラー", message=f"マスク画像の読込に失敗しました: {file}")
                return

        # マスク画像を使用してモザイク処理
        self.apply_mosaic(masks)

    def on_right_button_press(self, event):
        self.popup_menu.post(event.x_root, event.y_root)

    def on_selection_mode_change(self):
        mode = self.get_selection_mode()
        self.pen_manager.set_cursor_visible(mode == SelectionMode.PEN)
        self.move_pen_cursor_to_center()

    def on_filter_toolbar_click(self):
        self.mode = self.filter_mode_var.get()

    def calc_guideline_mosaic_size(self, store_mosaic_size=False):
        """某画像投稿サイトのガイドラインに則ったモザイクサイズを算出する"""
        assert self.image

        w, h = self.image.size
        m = max(4, math.ceil(max(w, h) / 100))
        self.guideline_mosaic_button.configure(value=m)

        if store_mosaic_size:
            self.mosaic_size_var.set(m)

    def on_close_window_without_save_image(self):
        self.save_window_geometry()
        root.destroy()

    def get_selection_mode(self) -> SelectionMode:
        return SelectionMode(self.selection_mode_var.get())

    def snap_coords_of_event(self, x, y, ceil=False):
        # canvas 上の表示座標 -> 実画像座標に変換してから snap する
        img_x = int(self.canvas.canvasx(x) / max(1e-6, self.scale / 100))
        img_y = int(self.canvas.canvasy(y) / max(1e-6, self.scale / 100))
        return self.snap_position(img_x, ceil), self.snap_position(img_y, ceil)

    def on_button_press(self, event):
        # マウスの左ボタンが押されたときに呼び出されるコールバック関数
        mode = self.get_selection_mode()

        if mode == SelectionMode.RECT:
            # 矩形選択
            self.start_x, self.start_y = self.snap_coords_of_event(event.x, event.y)

            # 矩形を描画するためのオブジェクトを作成する（表示座標に変換）
            sx = int(self.start_x * self.scale / 100)
            sy = int(self.start_y * self.scale / 100)
            self.rect = self.canvas.create_rectangle(sx, sy, sx, sy, fill="red", stipple='gray12', outline="red")
        elif mode == SelectionMode.PEN:
            # ペン塗りつぶし開始
            self.pen_manager.start_drawing(self.canvas.canvasx(event.x), self.canvas.canvasy(event.y))

    def on_mouse_move(self, event):
        # ボタンが押されていない状態でのマウス移動
        mode = self.get_selection_mode()

        if mode == SelectionMode.PEN:
            x, y = self.canvas.canvasx(event.x), self.canvas.canvasy(event.y)
            self.pen_manager.set_cursor_visible(True)
            self.pen_manager.move_cursor(x, y)
        else:
            self.pen_manager.set_cursor_visible(False)

    def on_move_press(self, event):
        # マウスの左ボタンが押されている状態でマウスが移動したときに呼び出されるコールバック関数
        mode = self.get_selection_mode()

        if mode == SelectionMode.RECT:
            # 矩形選択
            cur_x, cur_y = self.snap_coords_of_event(event.x, event.y, ceil=True)
            # 矩形を描画するためのオブジェクトを更新する（表示座標に変換）
            self.canvas.coords(self.rect, int(self.start_x * self.scale / 100), int(self.start_y * self.scale / 100), int(cur_x * self.scale / 100), int(cur_y * self.scale / 100))
        elif mode == SelectionMode.PEN:
            x, y = self.canvas.canvasx(event.x), self.canvas.canvasy(event.y)
            self.pen_manager.move_pen(x, y)

    def on_button_release(self, event):
        # マウスの左ボタンが離されたときに呼び出されるコールバック関数
        mode = self.get_selection_mode()

        if mode == SelectionMode.RECT:
            self.end_x, self.end_y = self.snap_coords_of_event(event.x, event.y, ceil=True)

            # 矩形を描画するためのオブジェクトを削除する
            self.canvas.delete(self.rect)

            # 矩形の範囲内の画素をモザイク化する
            self.apply_mosaic()
        elif mode == SelectionMode.PEN:
            self.pen_manager.move_pen(self.canvas.canvasx(event.x), self.canvas.canvasy(event.y))
            lines = self.pen_manager.drawing_finished()
            if lines:
                self.apply_pen_mosaic(lines)

    def apply_pen_mosaic(self, lines: list[int]):
        m = self.mosaic_size_var.get()
        p = self.pen_size_var.get()

        w, h = self.image.size
        hp = p // 2

        # ガウシアンブラーの半径
        # モザイクとペンサイズのどちらか小さいほうの半分を半径としてマスクをぼかす
        blur_radius = min(m, p) // 2

        # Undo用にペンの描画範囲が収まる矩形を取得する
        xs = sorted(map(int, lines[0::2]))
        ys = sorted(map(int, lines[1::2]))

        # マスクにぼかしを掛けるのでペンサイズと一緒にマージンを取る
        margin = hp + blur_radius * 2

        x1, x2 = max(0, xs[0] - margin), min(w - 1, xs[-1] + margin)
        y1, y2 = max(0, ys[0] - margin), min(h - 1, ys[-1] + margin)

        # Undoリストに追加する
        self.add_undo_list((x1, y1, x2, y2))

        # ペンの軌跡をマスクにする
        mask = Image.new("L", self.image.size, 0)
        draw = ImageDraw.Draw(mask)
        draw.line(lines, fill=255, width=p, joint="curve")
        # PillowのImageDrawモジュールは線分の両端の形状を指定できないため、線分の先端と末端に円を描画する
        draw.ellipse((lines[0] - hp, lines[1] - hp, lines[0] + hp, lines[1] + hp), fill=255)
        draw.ellipse(
            (lines[-2] - hp, lines[-1] - hp, lines[-2] + hp, lines[-1] + hp),
            fill=255,
        )

        # マスクをぼかす
        mask = mask.filter(ImageFilter.GaussianBlur(blur_radius))

        # 各種フィルタ処理
        filtered_image: Image

        if self.applist[self.mode] == 'ぼかし':
            filtered_image = self.image.filter(ImageFilter.GaussianBlur(m // 2))
        elif self.applist[self.mode] == '黒塗り':
            filtered_image = Image.new("RGB", (w, h), (0, 0, 0))
        else:  # モザイク
            filtered_image = self.image.resize((w // m, h // m), Image.Resampling.BICUBIC)
            filtered_image = filtered_image.resize((w, h), Image.Resampling.NEAREST)

        # 画像を更新する
        self.image.paste(filtered_image, (0, 0), mask)

        # キャンバスに表示する画像を更新する
        self.image_tk = ImageTk.PhotoImage(self.image)
        self.update_display_image()

    def apply_mosaic(self, mask_images: list[Image.Image] = None):
        if mask_images is None:
            # 矩形の範囲内の画素をモザイク化する
            # 矩形の左上と右下の座標を取得する
            x1, y1 = min(self.start_x, self.end_x), min(self.start_y, self.end_y)
            x2, y2 = max(self.start_x, self.end_x), max(self.start_y, self.end_y)
        else:
            # マスク画像が指定されている場合は画像全体をモザイク化し、マスクをかけて反映する
            x1, y1, x2, y2 = 0, 0, self.image.width, self.image.height

        # 座標を画像内に収める
        x1, x2 = [max(0, min(x, self.image.width )) for x in (x1, x2)]
        y1, y2 = [max(0, min(y, self.image.height)) for y in (y1, y2)]

        if (y1 == y2) or (x1 == x2):
            return

        # 範囲を取得する
        region = self.image.crop((x1, y1, x2, y2))

        # ぼかしの時
        if (self.applist[self.mode]=='ぼかし'):
            # 範囲をUndoリストに追加する
            self.add_undo_list((x1, y1, x2, y2))

            mosaic_size = self.mosaic_size_var.get()
            # ぼかしをかける
            width = (x2 - x1) // mosaic_size
            height = (y2 - y1) // mosaic_size
            if width == 0:
                width = 1
            if height == 0:
                height = 1
            region = region.resize((width, height), Image.Resampling.BICUBIC)
            region = region.resize((x2 - x1, y2 - y1), Image.Resampling.LANCZOS)

        # 黒塗りの時
        elif (self.applist[self.mode]=='黒塗り'):
            # 範囲をUndoリストに追加する
            self.add_undo_list((x1, y1, x2, y2))

            # 黒塗りにする
            region = Image.new("RGB", (x2 - x1, y2 - y1), (0, 0, 0))

        # 切抜の時
        elif (self.applist[self.mode]=='切抜'):
            # 全体をUndoリストに追加する
            self.add_undo_list()

            # キャンバスに表示する画像を更新する（選択範囲のみに切り抜き）
            self.image = region
            self.image_tk = ImageTk.PhotoImage(self.image)
            self.canvas.itemconfig(self.image_on_canvas, image=self.image_tk)
            self.image_width, self.image_height = self.image.size
            return

        # 縮小の時
        elif (self.applist[self.mode]=='縮小'):
            # 全体をUndoリストに追加する
            self.add_undo_list()

            # キャンバスに表示する画像を更新する（選択範囲のサイズに縮小）
            self.image =  self.image.resize((int(self.image_width*((y2 - y1)/self.image_height)),(y2 - y1)), Image.Resampling.BICUBIC)

            self.image_tk = ImageTk.PhotoImage(self.image)
            self.canvas.itemconfig(self.image_on_canvas, image=self.image_tk)
            self.image_width, self.image_height = self.image.size
            return

        # モザイク（小）～（大）の時
        else:
            # 範囲をUndoリストに追加する
            self.add_undo_list((x1, y1, x2, y2))

            mosaic_size = self.mosaic_size_var.get()
            # モザイクをかける
            width = (x2 - x1) // mosaic_size
            height = (y2 - y1) // mosaic_size
            if width == 0:
                width = 1
            if height == 0:
                height = 1
            region = region.resize((width, height), Image.Resampling.BICUBIC)
            region = region.resize((x2 - x1, y2 - y1), Image.Resampling.NEAREST)

        # 画像を更新する
        if mask_images is None:
            self.image.paste(region, (x1, y1, x2, y2))
        else:
            # マスク画像に従ってモザイク化する
            for mask in mask_images:
                self.image.paste(region, mask=mask.crop((x1, y1, x2, y2)))

        # キャンバスに表示する画像を更新する
        self.image_tk = ImageTk.PhotoImage(self.image)
        self.update_display_image()

    def on_mouse_wheel(self, event):
        # チルトホイールの時
        if event.state == 9:
            # 水平スクロールする
            self.canvas.xview_scroll(int(-0.05*event.delta), "units")
        # マウスホイールの時
        else:
            # 垂直スクロールする
            self.canvas.yview_scroll(int(-0.05*event.delta), "units")

    def on_shift_mouse_wheel(self, event):
        # 水平スクロールする
        self.canvas.xview_scroll(int(-0.05*event.delta), "units")

    def on_ctrl_mouse_wheel(self, event):
        # ペンのサイズを変更する
        self.pen_size_var.set(self.pen_size_var.get() + (5 if 0<=event.delta else -5))

    def on_ctrl_shift_mouse_wheel(self, event):
        # モザイクのサイズを変更する
        self.mosaic_size_var.set(self.mosaic_size_var.get() + (1 if 0<=event.delta else -1))

    def on_restore_draw_settings(self, event):
        # デフォルトの描画設定に戻す
        self.mosaic_size_var.set(DEFAULT_MOSAIC_SIZE)
        self.pen_size_var.set(DEFAULT_PEN_SIZE)
        self.filter_mode_var.set(DEFAULT_FILTER_MODE)
        self.on_filter_toolbar_click()
        self.selection_mode_var.set(DEFAULT_SELECTION_MODE)
        self.on_selection_mode_change()

    def read_jpeg_comment(file_path):
        img = Image.open(file_path)

    def save_image(self):
        #フォルダ指定モードかつ、モザイク処理の履歴がなにもない時、保存せずスキップする
        if len(self.image_file_list) > 0 and len(self.undo_list) == 0: return

        # 画像を保存する
        #出力先の親フォルダが存在しない場合は作成する
        self.create_folder_if_not_exist(os.path.dirname(self.write_path))
        image_format = self.image.format.lower() if self.image.format else 'unknown'
        if (image_format == 'jpeg') or (image_format == 'webp') or (image_format == 'avif'):
            #JPEG/webp/avif形式ならexif情報をそのままコピーして保存
            exif_data = self.image.info.get('exif', b'')
            # EXIFデータが存在する場合のみロードする
            if exif_data:
                exif_dict = piexif.load(exif_data)
            else:
                exif_dict = {}
            exif_bytes = piexif.dump(exif_dict)
            if (image_format == 'webp'):
                self.image.save(self.write_path, exif=exif_bytes, lossless=self.lossless)
            else:
                self.image.save(self.write_path, exif=exif_bytes)
        else:
            #それ以外なら（今のところpngのみを想定）メタデータからそれっぽいのを抽出して保存
            metadata = PngInfo()
            if self.parameters != '':
                metadata.add_text("parameters", self.parameters)
            if self.prompt != '':
                metadata.add_text("prompt", self.prompt)
            if self.workflow != '':
                metadata.add_text("workflow", self.workflow)
            self.image.save(self.write_path, pnginfo=metadata)

    def click_close(self):
        self.save_image()
        self.save_window_geometry()
        root.destroy()

    def snap_position(self, x, ceil=False):
        # モード別のグリッドのステップサイズを取得する
        step = self.mosaic_size_var.get()

        # グリッドに吸着
        x = int(x)
        m = x % step

        if m != 0:
            x -= m
            if ceil:
                x += step

        return x

    def update_display_image(self):
        """表示用の ImageTk.PhotoImage を作成してキャンバスに反映する"""
        if not hasattr(self, 'image'):
            return
        # 表示用にリサイズ
        if self.scale == 100:
            disp = self.image
        else:
            w = max(1, int(self.image_width * self.scale / 100))
            h = max(1, int(self.image_height * self.scale / 100))
            disp = self.image.resize((w, h), Image.Resampling.BICUBIC)

        self.display_image = disp
        self.image_tk = ImageTk.PhotoImage(disp)

        # キャンバスに反映
        if getattr(self, 'canvas', None) and getattr(self, 'image_on_canvas', None):
            try:
                self.canvas.itemconfig(self.image_on_canvas, image=self.image_tk)
                # スクロール領域を更新
                self.canvas.configure(scrollregion=(0, 0, disp.width, disp.height))
            except Exception:
                pass

        # ラベル更新
        if hasattr(self, 'zoom_label_var'):
            try:
                self.zoom_label_var.set(f"{int(self.scale)}%")
            except Exception:
                pass

    def set_scale(self, new_scale: int) -> None:
        """表示倍率を設定して表示を更新する"""
        old = getattr(self, 'scale', SCALE_DEFAULT)
        new = max(SCALE_MIN, min(SCALE_MAX, new_scale))
        self.scale = new
        # 表示イメージを更新
        self.update_display_image()

        # ペン表示をスケールに合わせる（カーソル位置を画像座標で再算出して再配置）
        if getattr(self, 'pen_manager', None):
            try:
                old_disp = self.pen_manager.cursor_pos
                img_x = old_disp[0] / max(1e-6, old)
                img_y = old_disp[1] / max(1e-6, old)
                self.pen_manager.set_scale(new)
                self.pen_manager.move_cursor(img_x * new, img_y * new)
            except Exception:
                try:
                    self.pen_manager.set_scale(new)
                except Exception:
                    pass

        # 選択矩形が描かれている場合も座標を更新
        if getattr(self, 'rect', None):
            try:
                coords = self.canvas.coords(self.rect)
                # coords は表示座標（old スケール）なので画像座標に変換して新表示座標にする
                img_coords = [c / max(1e-6, old) for c in coords]
                new_coords = [c * new for c in img_coords]
                self.canvas.coords(self.rect, *new_coords)
            except Exception:
                pass

    def zoom_in(self):
        if self.scale < SCALE_MAX:
            self.set_scale(self.scale + SCALE_STEP)

    def zoom_out(self):
        if self.scale > SCALE_MIN:
            self.set_scale(self.scale - SCALE_STEP)

    def reset_zoom(self):
        self.set_scale(SCALE_DEFAULT)

    def on_alt_mouse_wheel(self, event):
        # Alt + Wheel でズーム
        if event.delta > 0:
            self.zoom_in()
        else:
            self.zoom_out()

    def on_ctrl_keypress(self, event):
        """Ctrl+キー押下の汎用ハンドラ。keysym を見てズーム操作を行う。

        期待される keysym: 'plus', 'minus', 'equal', 'KP_Add', 'KP_Subtract'
        一部環境では 'plus' ではなく 'equal'（Shift 必要）などになるため複数判定する。
        """
        k = getattr(event, 'keysym', '')
        # Windows では '+' は Shift+'=' で送られることがある (keysym: 'plus' or 'equal')
        if k in ('plus', 'KP_Add'):
            self.zoom_in()
            return "break"
        if k in ('minus', 'KP_Subtract'):
            self.zoom_out()
            return "break"
        # '=' も Ctrl+Shift+= の形でズームインとして使われることがある
        if k in ('equal',):
            # 判定としては Shift が同時押しならインクリースに扱う
            state = getattr(event, 'state', 0)
            # Shift 修飾ビットが立っているか (プラットフォーム差あり) をざっくり判定
            if state & 0x1:
                self.zoom_in()
                return "break"

    # 実行ファイル(.pyか.exe)の位置にあるファイルをパス付きで取得する
    def find_data_file(self, filename):
        if getattr(sys, "frozen", False):
            datadir = os.path.dirname(sys.executable)
        else:
            datadir = os.path.dirname(__file__)
        return os.path.join(datadir, filename)

    def save_window_geometry(self, filename=WINDOW_GEOMETRY_FILE):
        geometry = root.geometry()
        with open(self.find_data_file(filename), 'w') as f:
            json.dump(geometry, f)

    def load_window_geometry(self, filename=WINDOW_GEOMETRY_FILE):
        if os.path.exists(self.find_data_file(filename)):
            with open(self.find_data_file(filename), 'r') as f:
                try:
                    geometry = json.load(f)
                    root.geometry(geometry)
                except:
                    print('ウィンドウサイズと位置情報の復元に失敗しました。')

    def save_drawing_settings(self, filename=DRAWING_SETTINGS_FILE):
        draw_settings = {
            'mosaic_size'    : self.mosaic_size_var.get(),
            'pen_size'       : self.pen_size_var.get(),
            'filter_mode'    : self.filter_mode_var.get(),
            'selection_mode' : self.selection_mode_var.get()}
        try:
            with open(self.find_data_file(filename), 'w') as f:
                json.dump(draw_settings, f)
        except:
            print('描画設定の保存に失敗しました。')
            tkinter.messagebox.showinfo(title='エラー', message='描画設定の保存に失敗しました。')

    def load_drawing_settings(self, filename=DRAWING_SETTINGS_FILE):        
        if os.path.exists(self.find_data_file(filename)):
            with open(self.find_data_file(filename), 'r') as f:
                try:
                    json_dic = json.load(f)
                    self.initial_draw_settings = DrawinfSettings(**json_dic)
                    return
                except:
                    print('描画設定の復元に失敗しました。')
        # 読み込めない場合は初期値を設定
        self.initial_draw_settings = DrawinfSettings()


def set_hidpi() -> None:
    if os.name == "nt":
        from ctypes import windll

        try:
            windll.shcore.SetProcessDpiAwareness(1)
        except Exception:
            pass  # this will fail on Windows Server and maybe early Windows


if __name__ == "__main__":
    set_hidpi()
    root = tk.Tk()
    app = App(master=root)
    root.protocol("WM_DELETE_WINDOW", app.click_close)
    app.mainloop()
