/**
 * This work is dedicated to the public domain under the CC0 1.0 Universal Public Domain Dedication.
 * For more information, see the LICENSE file.
 */

const EXCLUDE_LIST_KEY = 'beeneedleExcludeList';

// オプション画面の読み込み時に保存済みの設定を表示
document.addEventListener('DOMContentLoaded', () => {
  loadExcludeList();
});

// 保存ボタン
document.getElementById('save').addEventListener('click', () => {
  saveExcludeList();
});

// クリアボタン
document.getElementById('clear').addEventListener('click', () => {
  if (confirm('太字除外リストを全てクリアしますか？')) {
    document.getElementById('excludeList').value = '';
    saveExcludeList();
  }
});

// インポートボタン
document.getElementById('import').addEventListener('click', () => {
  document.getElementById('fileInput').click();
});

// ファイル選択時の処理
document.getElementById('fileInput').addEventListener('change', (event) => {
  const file = event.target.files[0];
  if (!file) return;

  if (!file.name.endsWith('.txt')) {
    showStatus('txtファイルを選択してください', 'error');
    event.target.value = ''; // ファイル選択をリセット
    return;
  }

  const reader = new FileReader();
  reader.onload = (e) => {
    const content = e.target.result;
    document.getElementById('excludeList').value = content;
    showStatus(`ファイル「${file.name}」を読み込みました。「保存」ボタンで反映してください。`, 'success');
    event.target.value = ''; // ファイル選択をリセット
  };
  reader.onerror = () => {
    showStatus('ファイルの読み込みに失敗しました', 'error');
    event.target.value = ''; // ファイル選択をリセット
  };
  reader.readAsText(file, 'UTF-8');
});

// エクスポートボタン
document.getElementById('export').addEventListener('click', () => {
  const textarea = document.getElementById('excludeList');
  const content = textarea.value;

  if (!content.trim()) {
    showStatus('エクスポートする内容がありません', 'error');
    return;
  }

  // BOM付きUTF-8で保存（Windowsのメモ帳でも文字化けしない）
  const bom = new Uint8Array([0xEF, 0xBB, 0xBF]);
  const blob = new Blob([bom, content], { type: 'text/plain;charset=utf-8' });
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  
  // ファイル名に日時を含める
  const now = new Date();
  const timestamp = now.toISOString().slice(0, 19).replace(/:/g, '-');
  a.href = url;
  a.download = `beeneedle_exclude_list_${timestamp}.txt`;
  
  document.body.appendChild(a);
  a.click();
  document.body.removeChild(a);
  URL.revokeObjectURL(url);

  showStatus('ファイルをエクスポートしました', 'success');
});

/**
 * 保存済みの除外リストを読み込んで表示
 */
function loadExcludeList() {
  chrome.storage.local.get([EXCLUDE_LIST_KEY], (result) => {
    if (chrome.runtime.lastError) {
      console.error('設定の読み込みエラー:', chrome.runtime.lastError);
      showStatus('設定の読み込みに失敗しました', 'error');
      return;
    }

    if (result[EXCLUDE_LIST_KEY] && Array.isArray(result[EXCLUDE_LIST_KEY])) {
      document.getElementById('excludeList').value = result[EXCLUDE_LIST_KEY].join('\n');
    }
  });
}

/**
 * 除外リストをストレージに保存（正規化前のテキストを保存）
 */
function saveExcludeList() {
  const textarea = document.getElementById('excludeList');
  const lines = textarea.value.split('\n');
  
  // 空行と#で始まるコメント行を除外し、前後の空白を削除（正規化はしない）
  const excludeList = lines
    .map(line => line.trim())
    .filter(line => line.length > 0 && !line.startsWith('#'));

  chrome.storage.local.set({ [EXCLUDE_LIST_KEY]: excludeList }, () => {
    if (chrome.runtime.lastError) {
      console.error('設定の保存エラー:', chrome.runtime.lastError);
      showStatus('設定の保存に失敗しました', 'error');
      return;
    }

    showStatus(`保存しました (${excludeList.length}件) - ページをリロードすると反映されます`, 'success');
  });
}

/**
 * ステータスメッセージを表示
 */
function showStatus(message, type) {
  const statusDiv = document.getElementById('status');
  statusDiv.textContent = message;
  statusDiv.className = `status ${type}`;
  statusDiv.style.display = 'block';

  // 3秒後に非表示
  setTimeout(() => {
    statusDiv.style.display = 'none';
  }, 3000);
}