/**
 * Prompter-E Main Application Logic
 */

// Initial Data Structure
const DEFAULT_CATEGORIES = [
    { id: 'quality', name: '品質' },
    { id: 'character', name: 'キャラクター' },
    { id: 'style', name: 'スタイル' },
    { id: 'background', name: '背景' }
];

const DEFAULT_TAGS = {
    'quality': [
        { ja: '高画質', en: 'masterpiece, best quality' },
        { ja: '超高画質', en: '8k wallpaper' },
        { ja: '高精細', en: 'highly detailed' },
        { ja: '低画質(Neg)', en: 'low quality, worst quality' }
    ],
    'character': [
        { ja: '1人の少女', en: '1girl' },
        { ja: 'ソロ', en: 'solo' },
        { ja: '笑顔', en: 'smile' },
        { ja: '長髪', en: 'long hair' }
    ],
    'style': [
        { ja: 'アニメ塗り', en: 'anime coloring' },
        { ja: 'フォトリアル', en: 'photorealistic' },
        { ja: '水彩', en: 'watercolor' }
    ],
    'background': [
        { ja: '青空', en: 'blue sky' },
        { ja: '室内', en: 'indoors' },
        { ja: 'シンプル', en: 'simple background' }
    ]
};

// Initial Presets per requirement: 3 Positive, 1 Negative
const DEFAULT_PRESETS = {
    positive: [
        { name: 'Default 1', content: '' },
        { name: 'Default 2', content: '' },
        { name: 'Default 3', content: '' }
    ],
    negative: [
        { name: 'Default Negative', content: 'low quality, worst quality, bad anatomy' }
    ]
};

// State Management
const STATE = {
    categories: safeJsonParse('pe_categories', DEFAULT_CATEGORIES),
    tags: safeJsonParse('pe_tags', DEFAULT_TAGS),
    selectedTags: safeJsonParse('pe_selected', {
        positive: [],
        negative: []
    }),
    presets: loadPresets(),
    currentMode: 'positive',
    currentCategory: null,
    editingType: null // 'category' or 'tag'
};

function safeJsonParse(key, defaultValue) {
    try {
        const item = localStorage.getItem(key);
        return item ? JSON.parse(item) : defaultValue;
    } catch (e) {
        console.warn(`Error parsing ${key} from localStorage, using default.`, e);
        return defaultValue;
    }
}

function loadPresets() {
    let raw;
    try {
        raw = JSON.parse(localStorage.getItem('pe_presets'));
    } catch (e) {
        console.warn('Error parsing presets, using default.', e);
        return DEFAULT_PRESETS;
    }
    // Migration or Default
    if (!raw) return DEFAULT_PRESETS;

    // Check if old format (array of strings/nulls)
    // We want to convert to objects
    const migrate = (arr, prefix) => {
        if (!Array.isArray(arr)) return [];
        return arr.map((item, idx) => {
            if (typeof item === 'string' || item === null) {
                return { name: `${prefix} ${idx + 1}`, content: item || '' };
            }
            return item; // Already object
        });
    };

    // If structure is different or old
    if (Array.isArray(raw.positive) && (typeof raw.positive[0] === 'string' || raw.positive[0] === null)) {
        return {
            positive: migrate(raw.positive, 'Preset'),
            negative: migrate(raw.negative, 'Neg Preset')
        };
    }

    return raw;
}

// DOM Elements
const els = {
    categoryList: document.getElementById('category-list'),
    tagPopup: document.getElementById('tag-popup'),
    tagGrid: document.getElementById('tag-grid'),
    popupTitle: document.getElementById('popup-title'),
    closeTagPopup: document.getElementById('close-tag-popup'),
    positiveOutput: document.getElementById('positive-output'),
    negativeOutput: document.getElementById('negative-output'),
    copyPositive: document.getElementById('copy-positive'),
    copyNegative: document.getElementById('copy-negative'),
    resetPositive: document.getElementById('reset-positive'),
    breakPositive: document.getElementById('break-positive'),
    resetNegative: document.getElementById('reset-negative'),
    modeRadios: document.getElementsByName('mode'),
    positivePresets: document.getElementById('positive-presets'),
    negativePresets: document.getElementById('negative-presets'),
    addPositivePresetBtn: document.getElementById('add-positive-preset'),
    addNegativePresetBtn: document.getElementById('add-negative-preset'),

    // Header Actions
    saveSettingsBtn: document.getElementById('save-settings-btn'),
    loadSettingsBtn: document.getElementById('load-settings-btn'),
    loadFileInput: document.getElementById('load-file-input'),

    // Editor Elements
    editorPopup: document.getElementById('editor-popup'),
    editorTitle: document.getElementById('editor-title'),
    closeEditorPopup: document.getElementById('close-editor-popup'),
    editorList: document.getElementById('editor-list'),
    addItemBtn: document.getElementById('add-item-btn'),
    saveEditorBtn: document.getElementById('save-editor-btn'),
    editCategoriesBtn: document.getElementById('edit-categories-btn'),
    editTagsBtn: document.getElementById('edit-tags-btn')
};

let sortableInstance = null;

// Initialization
function init() {
    // Ensure presets are valid arrays if something went wrong
    if (!STATE.presets.positive) STATE.presets.positive = [];
    if (!STATE.presets.negative) STATE.presets.negative = [];

    renderCategories();
    renderPresets();
    setupEventListeners();

    // Initial sync of outputs
    updateOutputsFromState();
}

function saveData() {
    localStorage.setItem('pe_categories', JSON.stringify(STATE.categories));
    localStorage.setItem('pe_tags', JSON.stringify(STATE.tags));
    localStorage.setItem('pe_selected', JSON.stringify(STATE.selectedTags));
    localStorage.setItem('pe_presets', JSON.stringify(STATE.presets));
}

// Rendering
function renderCategories() {
    els.categoryList.innerHTML = '';
    STATE.categories.forEach(cat => {
        const btn = document.createElement('button');
        btn.textContent = cat.name;
        btn.className = 'category-btn';
        btn.onclick = () => openTagPopup(cat.id, cat.name);
        els.categoryList.appendChild(btn);
    });
}

function openTagPopup(categoryId, categoryName) {
    STATE.currentCategory = categoryId;
    els.popupTitle.textContent = `${categoryName} (${STATE.currentMode === 'positive' ? 'Positive' : 'Negative'})`;
    renderTags(categoryId);
    els.tagPopup.classList.remove('hidden');
}

function renderTags(categoryId) {
    els.tagGrid.innerHTML = '';
    const tags = STATE.tags[categoryId] || [];
    const currentList = STATE.selectedTags[STATE.currentMode];

    tags.forEach(tag => {
        const btn = document.createElement('button');
        const selected = currentList.find(t => t.tag.en === tag.en);
        let baseClass = 'tag-btn';
        if (selected) {
            baseClass += selected.weight === 2 ? ' selected-double' : ' selected-single';
        }
        btn.className = baseClass;
        btn.textContent = tag.ja; // Show Japanese

        let clickTimeout;
        btn.addEventListener('click', (e) => {
            if (e.detail === 1) {
                clickTimeout = setTimeout(() => {
                    handleTagClick(tag, 1);
                    renderTags(categoryId);
                }, 200);
            }
        });
        btn.addEventListener('dblclick', () => {
            clearTimeout(clickTimeout);
            handleTagClick(tag, 2);
            renderTags(categoryId);
        });
        els.tagGrid.appendChild(btn);
    });
}

function handleTagClick(tag, clickType) {
    const list = STATE.selectedTags[STATE.currentMode];
    const existingIndex = list.findIndex(t => t.tag.en === tag.en);
    if (existingIndex > -1) {
        const currentWeight = list[existingIndex].weight;
        if (clickType === 2 && currentWeight === 1) {
            list[existingIndex].weight = 2;
        } else {
            list.splice(existingIndex, 1);
        }
    } else {
        list.push({ tag, weight: clickType });
    }
    saveData();
    updateOutputsFromState();
}

// Update Textareas from State
function updateOutputsFromState() {
    const pTags = formatTags(STATE.selectedTags.positive);
    els.positiveOutput.value = pTags;

    const nTags = formatTags(STATE.selectedTags.negative);
    els.negativeOutput.value = nTags;
}

function formatTags(selectedTags) {
    if (!selectedTags || selectedTags.length === 0) return '';

    let str = "";
    for (let i = 0; i < selectedTags.length; i++) {
        const item = selectedTags[i];
        const text = item.weight === 2 ? `(${item.tag.en})` : item.tag.en;

        if (i === 0) {
            str += text;
            continue;
        }

        const prevItem = selectedTags[i - 1];
        const prevText = prevItem.tag.en;

        if (text === 'BREAK') {
            // Before BREAK: comma, space, newline
            // But if previous was already BREAK, avoided doubling? 
            // Standard requirement: "tag, \nBREAK \n"
            str += ", \n" + text + " \n";
        } else {
            // Normal tag
            if (prevText === 'BREAK') {
                // If previous was BREAK, it ended with " \n" (no comma)
                str += text;
            } else {
                // Normal separator
                str += ", " + text;
            }
        }
    }
    return str;
}

// Update State from Textareas (Reverse Sync)
function updateStateFromOutput(type) {
    const text = type === 'positive' ? els.positiveOutput.value : els.negativeOutput.value;
    reconstructTagsFromText(text, type);
}

function reconstructTagsFromText(text, type) {
    STATE.selectedTags[type] = [];
    if (!text || text.trim() === '') {
        // If empty, just clear state
        // We do NOT call updateOutputsFromState here to avoid cursor jumping if user is typing
        saveData();
        return;
    }

    const items = text.split(/[,\n]+/).map(s => s.trim()).filter(s => s);
    const allTags = [];
    Object.values(STATE.tags).forEach(arr => allTags.push(...arr));

    items.forEach(item => {
        let cleanTag = item;
        let weight = 1;
        if (item.startsWith('(') && item.endsWith(')')) {
            cleanTag = item.slice(1, -1);
            weight = 2;
        }

        // Try to match exact known English tag
        const found = allTags.find(t => t.en.toLowerCase() === cleanTag.toLowerCase());
        if (found) {
            STATE.selectedTags[type].push({ tag: found, weight });
        } else {
            // New custom tag
            STATE.selectedTags[type].push({ tag: { ja: cleanTag, en: cleanTag }, weight });
        }
    });

    saveData();
    // Refresh tag view if open to show selection
    if (!els.tagPopup.classList.contains('hidden') && STATE.currentCategory) {
        renderTags(STATE.currentCategory);
    }
}


// Preset Logic
function renderPresets() {
    renderPresetList(els.positivePresets, STATE.presets.positive, 'positive');
    renderPresetList(els.negativePresets, STATE.presets.negative, 'negative');
}

function renderPresetList(container, presetsData, type) {
    container.innerHTML = '';
    presetsData.forEach((preset, index) => {
        const wrapper = document.createElement('div');
        wrapper.className = 'preset-item';

        // Name Input
        const nameInput = document.createElement('input');
        nameInput.type = 'text';
        nameInput.className = 'preset-name-input';
        nameInput.value = preset.name;
        nameInput.onchange = (e) => {
            preset.name = e.target.value;
            saveData();
        };

        // Save (Overwrite) Button
        const saveBtn = document.createElement('button');
        saveBtn.className = 'preset-action-btn preset-save-btn';
        saveBtn.textContent = '💾';
        saveBtn.title = 'Save current prompt to this preset';
        saveBtn.onclick = () => {
            const currentText = type === 'positive' ? els.positiveOutput.value : els.negativeOutput.value;
            preset.content = currentText;
            saveData();
            // Optional: visual feedback
            saveBtn.textContent = '✅';
            setTimeout(() => saveBtn.textContent = '💾', 1000);
        };

        // Load Button
        const loadBtn = document.createElement('button');
        loadBtn.className = 'preset-action-btn preset-load-btn';
        loadBtn.textContent = '▶️';
        loadBtn.title = 'Load this preset';
        loadBtn.onclick = () => {
            if (type === 'positive') els.positiveOutput.value = preset.content;
            else els.negativeOutput.value = preset.content;
            reconstructTagsFromText(preset.content, type);
        };

        // Delete Button
        const delBtn = document.createElement('button');
        delBtn.className = 'preset-action-btn preset-del-btn';
        delBtn.textContent = '🗑️';
        delBtn.title = 'Delete preset';
        delBtn.onclick = () => {
            if (confirm('Delete this preset?')) {
                STATE.presets[type].splice(index, 1);
                saveData();
                renderPresets();
            }
        };

        wrapper.appendChild(nameInput);
        wrapper.appendChild(saveBtn);
        wrapper.appendChild(loadBtn);
        wrapper.appendChild(delBtn);
        container.appendChild(wrapper);
    });
}

function addPreset(type) {
    const newPreset = { name: `New Preset`, content: '' };
    STATE.presets[type].push(newPreset);
    saveData();
    renderPresets();
}

// File I/O
async function exportSettings() {
    const dataStr = JSON.stringify(STATE, null, 2);

    // Try File System Access API
    if (window.showSaveFilePicker) {
        try {
            const handle = await window.showSaveFilePicker({
                suggestedName: 'settings.json',
                types: [{
                    description: 'JSON Files',
                    accept: { 'application/json': ['.json'] },
                }],
            });
            const writable = await handle.createWritable();
            await writable.write(dataStr);
            await writable.close();
            return; // Success
        } catch (err) {
            // User cancelled or error, fall back to blob if not cancellation
            if (err.name !== 'AbortError') {
                console.warn('File Save Error:', err);
                downloadBlob(dataStr);
            }
            return;
        }
    }

    // Fallback
    downloadBlob(dataStr);
}

function downloadBlob(content) {
    const blob = new Blob([content], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'settings.json';
    document.body.appendChild(a);
    a.click();
    setTimeout(() => {
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }, 0);
}

function importSettings(e) {
    const file = e.target.files[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = (event) => {
        try {
            const data = JSON.parse(event.target.result);
            // Basic validation
            if (data.categories && data.tags) {
                // Update State
                Object.assign(STATE, data);
                saveData();

                // Re-render
                renderCategories();
                renderPresets();
                updateOutputsFromState();

                alert('Settings loaded successfully!');
            } else {
                throw new Error('Invalid format');
            }
        } catch (err) {
            alert('Error loading file: ' + err.message);
        }
        // Reset input
        e.target.value = '';
    };
    reader.readAsText(file);
}

// Editor Logic (reused with tweaks)
function openEditor(type) {
    STATE.editingType = type;
    els.editorPopup.classList.remove('hidden');
    els.editorList.innerHTML = '';

    if (type === 'category') {
        els.editorTitle.textContent = 'Edit Categories';
        renderEditorItems(STATE.categories, type);
    } else if (type === 'tag') {
        if (!STATE.currentCategory && STATE.categories.length > 0) {
            STATE.currentCategory = STATE.categories[0].id;
        }
        if (!STATE.currentCategory) {
            alert('No categories available.');
            return;
        }

        const cat = STATE.categories.find(c => c.id === STATE.currentCategory);
        els.editorTitle.textContent = `Edit Tags: ${cat ? cat.name : 'Unknown'}`;
        if (!STATE.tags[STATE.currentCategory]) STATE.tags[STATE.currentCategory] = [];
        renderEditorItems(STATE.tags[STATE.currentCategory], type);
    }

    if (sortableInstance) sortableInstance.destroy();
    sortableInstance = new Sortable(els.editorList, {
        handle: '.handle',
        animation: 150
    });
}

function renderEditorItems(items, type) {
    els.editorList.innerHTML = '';
    items.forEach(item => {
        const row = createEditorRow(item, type);
        els.editorList.appendChild(row);
    });
}

function createEditorRow(item, type) {
    const row = document.createElement('div');
    row.className = 'editor-item';
    row.innerHTML = `<span class="handle">☰</span>`;

    const inputs = document.createElement('div');
    inputs.className = 'editor-row-inputs';

    if (type === 'category') {
        inputs.innerHTML = `
            <input type="text" class="editor-input" placeholder="Display Name" value="${item.name || ''}" data-key="name">
            <input type="text" class="editor-input" placeholder="ID (Unique)" value="${item.id || ''}" data-key="id" ${item.id ? 'disabled' : ''}>
        `;
    } else {
        inputs.innerHTML = `
            <input type="text" class="editor-input" placeholder="Japanese" value="${item.ja || ''}" data-key="ja">
            <input type="text" class="editor-input" placeholder="English Tag" value="${item.en || ''}" data-key="en">
        `;
    }
    row.appendChild(inputs);

    const delBtn = document.createElement('button');
    delBtn.className = 'delete-btn';
    delBtn.textContent = '🗑️';
    delBtn.onclick = () => row.remove();
    row.appendChild(delBtn);

    return row;
}

function addNewItem() {
    let newItem = {};
    if (STATE.editingType === 'category') {
        newItem = { name: '', id: 'cat_' + Date.now() };
    } else {
        newItem = { ja: '', en: '' };
    }
    const row = createEditorRow(newItem, STATE.editingType);
    els.editorList.appendChild(row);
}

function saveEditor() {
    const rows = Array.from(els.editorList.children);
    const newItems = [];

    rows.forEach(row => {
        const inputs = row.querySelectorAll('input');
        const item = {};
        inputs.forEach(input => {
            const key = input.getAttribute('data-key');
            if (key) item[key] = input.value;
        });

        if (STATE.editingType === 'category') {
            if (item.name && item.id) newItems.push(item);
        } else {
            if (item.ja && item.en) newItems.push(item);
        }
    });

    if (STATE.editingType === 'category') {
        STATE.categories = newItems;
        renderCategories();
        // If current category deleted, reset
        if (!STATE.categories.find(c => c.id === STATE.currentCategory)) {
            STATE.currentCategory = null;
        }
    } else {
        STATE.tags[STATE.currentCategory] = newItems;
        if (!els.tagPopup.classList.contains('hidden')) {
            renderTags(STATE.currentCategory);
        }
    }

    saveData();
    els.editorPopup.classList.add('hidden');
}

function setupEventListeners() {
    // Popup
    els.closeTagPopup.onclick = () => els.tagPopup.classList.add('hidden');
    els.tagPopup.onclick = (e) => {
        if (e.target === els.tagPopup) els.tagPopup.classList.add('hidden');
    };

    // Output Actions
    els.copyPositive.onclick = () => navigator.clipboard.writeText(els.positiveOutput.value);
    els.copyNegative.onclick = () => navigator.clipboard.writeText(els.negativeOutput.value);

    els.breakPositive.onclick = () => {
        const list = STATE.selectedTags.positive;
        // Add pseudo-tag for BREAK
        list.push({ tag: { ja: 'BREAK', en: 'BREAK' }, weight: 1 });

        saveData();
        updateOutputsFromState();
        // Focus back to textarea usually desirable, or just button
        // els.positiveOutput.focus(); 
    };

    els.resetPositive.onclick = () => {
        els.positiveOutput.value = '';
        updateStateFromOutput('positive');
    };
    els.resetNegative.onclick = () => {
        els.negativeOutput.value = '';
        updateStateFromOutput('negative');
    };

    // Textarea Editing (Debounced)
    let timeoutP, timeoutN;
    els.positiveOutput.addEventListener('input', () => {
        clearTimeout(timeoutP);
        timeoutP = setTimeout(() => updateStateFromOutput('positive'), 500);
    });
    els.negativeOutput.addEventListener('input', () => {
        clearTimeout(timeoutN);
        timeoutN = setTimeout(() => updateStateFromOutput('negative'), 500);
    });

    // Mode Switch
    Array.from(els.modeRadios).forEach(radio => {
        radio.addEventListener('change', (e) => {
            STATE.currentMode = e.target.value;
            if (!els.tagPopup.classList.contains('hidden') && STATE.currentCategory) {
                const cat = STATE.categories.find(c => c.id === STATE.currentCategory);
                if (cat) {
                    els.popupTitle.textContent = `${cat.name} (${STATE.currentMode === 'positive' ? 'Positive' : 'Negative'})`;
                    renderTags(STATE.currentCategory);
                }
            }
        });
    });

    // Preset Buttons
    els.addPositivePresetBtn.onclick = () => addPreset('positive');
    els.addNegativePresetBtn.onclick = () => addPreset('negative');

    // Header Actions
    els.saveSettingsBtn.onclick = exportSettings;
    els.loadSettingsBtn.onclick = () => els.loadFileInput.click();
    els.loadFileInput.onchange = importSettings;

    // Editor Handlers
    els.editCategoriesBtn.onclick = () => openEditor('category');
    els.editTagsBtn.onclick = () => openEditor('tag');
    els.closeEditorPopup.onclick = () => els.editorPopup.classList.add('hidden');
    els.addItemBtn.onclick = addNewItem;
    els.saveEditorBtn.onclick = saveEditor;
}

// Start
init();
